from typing import Optional

import requests
import datetime
import logging
import os

# https://wiki.yandex-team.ru/security/ssl/sslclientfix
os.environ["REQUESTS_CA_BUNDLE"] = "/etc/ssl/certs/ca-certificates.crt"


def make_reliable(f):
    def wrapper(*args, **kwargs):
        reliability = 3
        for i in range(reliability):
            try:
                return f(*args, **kwargs)
            except Exception:
                continue
        return f(*args, **kwargs)

    return wrapper


class Gap(object):
    """
    This Class has methods for working with two different API's:
        1. staff/gap API
        2. internal corp calendar API (prod)

    """

    base_url = ""
    token = ""
    base_calendar_url = ""
    CALENDAR_YATEAM_TVM_ID = "2011072"

    def __init__(
        self,
        base_url,
        token,
        base_calendar_url,
        calendar_tvm_service_ticket=None,
        gap_tvm_service_ticket=None,
    ):
        """
        base_calendar_url should be provided with calendar_tvm_service_ticket after 19.02.2020.
        More info here: https://clubs.at.yandex-team.ru/calendar/2838

        This arguments required for this methods:
         - get_holidays
         - get_holidays_info_from_calendar [DEPRECATED]
         - is_workday [REFACTOR required after 04.03.2020]
         - get_closest_workday
         - add_delay_if_holidays

        """

        self.base_url = base_url.strip()
        self.token = token.strip()
        self.base_calendar_url = base_calendar_url.strip()
        self.base_holidays = {
            "31.12",
            "01.01",
            "02.01",
            "03.01",
            "04.01",
            "05.01",
            "06.01",
            "07.01",
            "23.02",
            "08.03",
            "01.05",
            "02.05",
            "09.05",
            "12.06",
            "04.11",
        }
        self.logger = logging.getLogger(__name__)
        self.__calendar_tvm_service_ticket = calendar_tvm_service_ticket

        # TODO: For future, then gap API will start using TVM
        self.__gap_tvm_service_ticket = gap_tvm_service_ticket

    def get_availability(
        self, date_from, date_to, logins, working_hour_from=10, working_hour_to=19
    ):
        payload = {
            "date_from": date_from,
            "date_to": date_to,
            "l": logins,
            "working_hour_from": working_hour_from,
            "working_hour_to": working_hour_to,
            "use_work_in_absence": 1,
            "use_calendar": 1,
        }

        url = self.base_url + "/availability"
        headers = {
            "content-type": "application/json",
            "Authorization": "OAuth " + self.token,
        }
        r = requests.get(url, params=payload, headers=headers)
        json_data = r.json()
        return json_data.get("persons", {})

    @make_reliable
    def get_working_today(self, logins):
        payload = {"l": logins}

        url = "%s%s" % (self.base_url, "/export_gaps")
        headers = {
            "content-type": "application/json",
            "Authorization": "OAuth %s" % self.token,
        }
        r = requests.get(url, params=payload, headers=headers)
        json_data = r.json()
        result = json_data.get("persons", {})
        if result:
            return list(result.values())[0][0]["work_in_absence"]
        return True

    @staticmethod
    def _check_date_validity(date, date_format="%Y-%m-%d"):
        result = False
        try:
            if datetime.datetime.strptime(date, date_format):
                result = True
        except ValueError:
            result = False

        return result

    def get_holidays(
        self, start_date: str, end_date: str = None, holidays_country_code: str = "rus"
    ) -> list:
        """
        Uses this base calendar url: "https://calendar-api.tools.yandex.net/internal"
        https://wiki.yandex-team.ru/Calendar/api/new-web/#get-holidays

        You should provide TVM service ticket for this API via instance constructor (checkout __init__()).


        :param start_date: Start date as string with format "%Y-%m-%d".
        :type start_date: str
        :param end_date: End date as string with format "%Y-%m-%d".
        :type end_date: str
        :param holidays_country_code: Country code for which to look for holidays. Defaults to 'rus'
        :type holidays_country_code: str
        :return: List with holidays if they exist. Each holiday is represented as dict.
            Holidays have three types: weekday, weekend, holiday.
        :rtype: list
        """
        if end_date is None:
            end_date = start_date

        if not self._check_date_validity(start_date) or not self._check_date_validity(
            end_date
        ):
            raise ValueError("Date provided in wrong format!")

        url = "/".join([self.base_calendar_url, "get-holidays"])
        payload = (
            ("from", start_date),
            ("to", end_date),
            ("for", holidays_country_code),
        )

        headers = {"X-Ya-Service-Ticket": self.__calendar_tvm_service_ticket}
        result = requests.get(url=url, params=payload, headers=headers)
        result_json = result.json()

        return result_json["holidays"]

    # TODO: Remove this after some delay, reason: deprecated
    def get_holidays_info_from_calendar(
        self,
        start_date,
        end_date=None,
        api_method="get-holidays",
        holidays_country_code="rus",
    ):
        # type: (str, str, str, str) -> bool
        """
        DEPRECATED: Use get_holidays. This function also contains a bug.

        If between start_date and end_date exist holiday, then return False.
        Otherwise - True.

        :param start_date:
        :type start_date:
        :param end_date:
        :type end_date:
        :param api_method:
        :type api_method:
        :param holidays_country_code:
        :type holidays_country_code:
        :return:
        :rtype:
        """

        logging.critical("WARNING! This Function is DEPRECATED!")

        if not self._check_date_validity(start_date):
            raise ValueError

        if end_date is None:
            end_date = start_date

        url = "/".join([self.base_calendar_url, api_method])
        payload = (
            ("from", start_date),
            ("to", end_date),
            ("for", holidays_country_code),
        )

        result = requests.get(url=url, params=payload)
        result_json = result.json()

        # Default to weekday, if API failed to return answer
        day_type = "weekday"

        if result_json.get("holidays"):
            # BUG HERE: should check all entries
            day_type = result_json["holidays"][0]["type"]
        # else:
        #     # Add logging + maybe raise exception + add test
        #     # logger.error("API returned wrong result, debug needed.
        #     # Answer in json: {}".format(result_json))

        if day_type == "weekday":
            return True
        else:
            return False

    def is_workday(self, date=None):
        # type: (Optional[str]) -> bool
        """
        Check, today is workday or not (if date not provided).
        Otherwise - check if provided date is workday (type: weekday)

        :param date: Date for check
        :type date: str
        :return: True or False
        :rtype: bool
        :raises: ValueError if date does not match format %Y-%m-%d
        :raises: TypeError if date is not str or str
        """

        if date is None:
            # None -> str (with current date)
            date = datetime.date.today().strftime("%Y-%m-%d")
        # For unicode_literals compatibility
        elif isinstance(date, str) or isinstance(date, str):
            # Check, if date is in right format.
            # If not, ValueError will be raised
            _ = datetime.datetime.strptime(date, "%Y-%m-%d")
        else:
            raise TypeError("date must be 'str' type and" " in '%Y-%m-%d' timeformat!")

        try:
            if self.__calendar_tvm_service_ticket is None:
                return self.get_holidays_info_from_calendar(start_date=date)

            holidays = self.get_holidays(start_date=date)
            for day in holidays:
                if day["type"] in ("weekend", "holiday"):
                    return False
            return True

        except Exception:

            failover = datetime.datetime.strptime(date, "%Y-%m-%d")
            if (
                failover.strftime("%d.%m") in self.base_holidays
                or failover.weekday() > 4
            ):
                return False

        return True

    def user_gaps_find(self, login, date_from=None, date_to=None):
        # type: (str, Optional[str], Optional[str]) -> Optional[list]
        """
        Find all user gaps for time interval. Via /gap-api/api/gaps_find/

        :param login: User login
        :type login: str
        :param date_from: Interval starting date in %Y-%m-%d format,
         defaults to *None* (current date)
        :type date_from: str
        :param date_to: Interval ending date in %Y-%m-%d format,
         defaults to *None* (current date + timedelta=24h)
        :type date_to: str
        :return: List with finded gaps (represented as dicts)
        :rtype: list
        """

        if date_from is None:
            date_from = datetime.date.today()
            date_from = date_from.strftime("%Y-%m-%d")

        if date_to is None:
            date_to = datetime.datetime.strptime(
                date_from, "%Y-%m-%d"
            ) + datetime.timedelta(hours=24)
            date_to = date_to.strftime("%Y-%m-%d")

        url_method = "gaps_find/"
        url = "/".join([self.base_url, url_method])
        payload = {"person_login": login, "date_from": date_from, "date_to": date_to}

        headers = {
            "content-type": "application/json",
            "Authorization": "OAuth %s" % self.token,
        }

        result = requests.get(url=url, params=payload, headers=headers)

        try:
            result_json = result.json()
        except Exception:
            self.logger.info(
                "Response status code: {0};"
                "Response text: {1}".format(result.status_code, result.text)
            )
            self.logger.info(
                "Login={0}; date_from={1}; date_to={2}".format(
                    login, date_from, date_to
                )
            )
            self.logger.exception(
                "Bad answer recieved, exception raised"
                " while trying to parse answer as JSON!"
            )
            result_json = dict()

        return result_json.get("gaps")

    def get_date_then_user_available(self, login, date=None):
        # type: (str, str) -> str
        """
        Check, is user available today. If yes, then return current date,
        otherwise - date then user will be available.

        :param login: User login
        :type login: str
        :param date: Base date for check in %Y-%m-%d format
        :type date: str
        :return: Date in %Y-%m-%d format
        :rtype: str
        """

        gaps = self.user_gaps_find(login=login, date_from=date)
        result = date

        if gaps:
            recent_gap = gaps[-1]
            work_in_absence = recent_gap.get("work_in_absence")
            gap_date_from = recent_gap.get("date_from").split("T")[0]

            if not work_in_absence and self.date_string_to_datetime_date(
                date
            ) >= self.date_string_to_datetime_date(gap_date_from):
                result = recent_gap.get("date_to")

        if "T" in result:
            result = result.split("T")[0]

        return result

    def get_closest_workday(self, base_date, result_in_datetime=False):
        # type: (str, bool) -> Union[str, datetime.datetime]
        """
        Iterate over + timedelta(hours=24) until not workday

        :param base_date: Start date, from which to find closest
        :type base_date: str
        :param result_in_datetime: Return result as datetime.datetime
         if *True*. Otherwise return str
        :type result_in_datetime: bool
        :return: Time in str string or datetime.datetime
        :rtype: str, datetime.datetime
        """

        base_date = str(base_date)
        date_is_workday = self.is_workday(date=base_date)

        result = base_date

        if not date_is_workday:
            while not date_is_workday:
                base_date = (
                    datetime.datetime.strptime(base_date, "%Y-%m-%d")
                    + datetime.timedelta(hours=24)
                ).strftime("%Y-%m-%d")
                date_is_workday = self.is_workday(date=base_date)
                result = base_date

        result = str(result)

        if result_in_datetime:
            result = datetime.datetime.strptime(result, "%Y-%m-%d")

        return result

    def add_delay_if_holidays(self, base_date, delay=24):
        # type: (str, int) -> str
        """
        If date is not workday, skip to closest workday + 24h.
        If it is Friday, scroll to Tuesday

        :param base_date:
        :type base_date:
        :param delay:
        :type delay:
        :return:
        :rtype:
        """
        base_date = str(base_date)

        if not self.is_workday(base_date):
            closest_workday = self.get_closest_workday(base_date)
            result = self._add_timedelta_to_date_string(
                base_date=closest_workday, delta_in_hours=delay
            )
            if not self.is_workday(result):
                result = self.add_delay_if_holidays(result)
        else:
            result = base_date

            result = str(result)

        return result

    @staticmethod
    def _add_timedelta_to_date_string(base_date, delta_in_hours):
        # type: (str, int) -> str
        """
        Add timedelta in hours to date represented as string.

        :param base_date: Base date
        :type base_date: str
        :param delta_in_hours: Timedelta in hours
        :type delta_in_hours: int
        :return: Result date
        :rtype: str
        """
        base_date = str(base_date)
        result = (
            datetime.datetime.strptime(base_date, "%Y-%m-%d")
            + datetime.timedelta(hours=delta_in_hours)
        ).strftime("%Y-%m-%d")
        result = str(result)

        return result

    def check_user_available_on_date(self, base_date, login):
        # type: (str, str) -> bool
        """
        Check does user available at provided date.

        :param base_date: Date for check
        :type base_date: str
        :param login: User login
        :type login: str
        :return: True or False
        :rtype: bool
        """
        base_date = str(base_date)

        try:
            user_availability_date = self.get_date_then_user_available(
                login=login, date=base_date
            )
        except Exception as exc:
            self.logger.error("Exception raised: {}".format(exc))
            user_availability_date = None
            self.logger.exception("Failed to get info from Gap lookup!")

        if user_availability_date == base_date:
            result = True
        else:
            result = False

        return result

    @staticmethod
    def date_string_to_datetime_date(date_string):
        # type: (str) -> datetime.date
        """
        Transform date in string format into datetime.date

        :param date_string:
        :type date_string:
        :return:
        :rtype:
        """

        result = datetime.datetime.date(
            datetime.datetime.strptime(date_string, "%Y-%m-%d")
        )
        return result
