# -*- coding: utf-8 -*-
from __future__ import unicode_literals
import requests
from typing import Optional  # noqa
import yaml
import logging

logger = logging.getLogger(__name__)


class Host(object):
    """
    Abstraction for Host, to fetch host responsibles.

    Have two methods:

        1. get_responsible_from_ro - fetch info from Admin interface
           (same as in Golem)
        2. get_responsible_from_conductor - fetch info from Conductor
           (contain all hostname group admins).

    """

    def __init__(self, hostname):
        # type: (str) -> None
        """

        :param hostname: Hostname FQDN
        :type hostname: str
        """

        self.hostname = hostname

    @staticmethod
    def fetch_data_via_get(url, params=None, timeout=None):
        # type: (str, Optional[dict], Optional[int]) -> Optional[requests.models.Response]
        """
        Base method, which calls requests.get function.

        :param url: API url to access
        :type url: str
        :param params: Optional, defaults to *None*.
         Parameters to be passed as params to requests.get
        :type params: dict
        :param timeout: Optional, defaults to *None*.
         Timeout to be passed as argument to requests.get
        :type timeout: int
        :return: Requests response object
        :rtype: requests.models.Response
        """

        args = {"url": url}

        if isinstance(params, dict) and params:
            args["params"] = params

        if isinstance(timeout, int):
            args["timeout"] = timeout

        try:
            result = requests.get(**args)
        except Exception:
            logger.exception("Failed to fetch data! Args=%s", args)
            result = None

        return result

    def get_responsible_from_ro(self):
        # type: () -> Optional[list]
        """
        Fetch info from ro.admin

        :return: List with admins logins or None
        :rtype:  list
        """

        url = "http://ro.admin.yandex-team.ru/api/get_host_resp.sbml"
        result = self.fetch_data_via_get(
            url=url, params={"hostname": self.hostname}, timeout=5
        )

        if result is not None:
            if result.text:
                result = result.text.strip().split(",")

            if len(result) == 1 and not result[0]:
                logger.warning(
                    "Responsible not found, returning None." "Response result: %s",
                    result,
                )
                result = None

        return result

    def get_responsible_from_conductor(self):
        # type: () -> Optional[list]
        """
        Fetch info from conductor via /api/hosts in yaml format.

        :return: List with admins logins or None
        :rtype: list
        """

        url = "https://c.yandex-team.ru/api/hosts/{}?format=yaml".format(self.hostname)
        result = self.fetch_data_via_get(url=url, timeout=5)

        if result is not None:
            result = yaml.safe_load(result.text.replace(" :", " "))[0]

            if isinstance(result, dict):
                result = result.get("admins")
            else:
                logger.warning(
                    "Responsible is not found, returning None." "Response result: %s",
                    result,
                )
                result = None

        return result

    def get_responsible(self):
        # type: () -> Optional[str]
        """
        Returns only one responsible login.
        First method is get_responsible_from_ro,
        fallback is get_responsible_from_conductor

        :return: Responsible admin login or None if not found
        :rtype: str
        """
        responsible = self.get_responsible_from_ro()

        if responsible is None or not responsible:
            responsible = self.get_responsible_from_conductor()
            logger.warning(
                "Responsible not found via ro.admin," " using conductor info..."
            )

        if isinstance(responsible, list):
            responsible = responsible[0]

        return responsible
