from requests import get, post


class Idm(object):
    base_url = "https://idm-api.yandex-team.ru/"
    token = ""

    def __init__(self, base_url, token):
        self.base_url = base_url.strip()
        self.token = token.strip()
        self.base_headers = {
            "content-type": "application/json",
            "Authorization": "OAuth " + self.token,
        }

    def get_rules_by_system(self, system, state="granted"):
        # type: (str) -> dict
        """
        Return list of dict's of idm roles in system that has idm role in choosen system.

        :param system: idm slug of system
        :type system: str or str
        :param state: state of role in idm system
        defaults: granted (aka active)
        all types of role u can find here https://wiki.yandex-team.ru/Intranet/idm/states/
        :type state: str or str
        :return: list of dict's of roles
        :rtype: list of dict's

        """
        route = (
            "api/v1/roles/?format=json&system=" + str(system) + "&state=" + str(state)
        )
        url = self.base_url + route
        payload = {"format": "json", "system": system, "state": state}
        r = get(url, params=payload, headers=self.base_headers)
        json_data = r.json()
        rules = json_data.get("objects")
        meta = json_data.get("meta")  # type: dict

        while meta["next"] is not None:
            route = meta["next"]
            url = self.base_url + route
            r_next = get(url, params=payload, headers=self.base_headers)
            json_data = r_next.json()
            transit = json_data.get("objects")
            rules = rules + transit
            meta = json_data.get("meta")

        return rules

    def get_logins_by_system(self, system, state="granted"):
        # type: (str) -> list
        """
        Return logins that has idm role in choosen system.


        :param system: idm slug of system
        :type system: str or str
        :param state: state of role in idm system
        defaults: granted (aka active)
        all types of role u can find here https://wiki.yandex-team.ru/Intranet/idm/states/
        :type state: str or str
        :return: list of logins
        :rtype: list
        """
        rules = self.get_rules_by_system(system, state)
        logins = list()

        for rule in rules:
            login = rule["user"]["username"]
            logins.append(login)

        return logins

    def request_role(self, system, path, user=None, group=None):
        # type: (str) -> dict
        """
        Return user object in success and None otherwise.

        :param system: idm slug of system
        :type system: str
        :param path: state of role in idm system
        :type path: str
        :param user: user login
        :type user: str
        :param group: group id
        :type group: int
        :return: user object
        :rtype: dict

        """
        data = {
            "system": system,
            "path": path,
            "silent": True,
        }
        if user:
            data["user"] = user
        elif group:
            data["group"] = group
        else:
            return None
        url = self.base_url + "/rolerequests/"
        res = post(url, json=data, headers=self.base_headers)
        if res.status_code == 500:
            return None
        return res.json()
