import logging
import os.path
from enum import IntEnum
from typing import Optional, Dict, Union
from urllib.parse import urlparse, urlunparse

from requests import get, post, Response

from security.yaseclib.abc import Abc
from security.yaseclib.common.utils import DictVal, safe_func

logger = logging.getLogger()


class RoleId(IntEnum):
    # ABC
    HEAD_OF_PRODUCT = 1
    DEVELOPER = 8
    FUNCTIONAL_TESTER = 19
    PRODUCT_MANAGER = 35

    SYSTEM_ADMINISTRATOR = 16
    DB_ADMINISTRATOR = 17

    TVM_MANAGER = 630
    TVM_SSH_USER = 631

    RESPONSIBLE_FOR_CERT = 960
    PERSONAL_DATA_RESPONSIBLE = 4393

    DUTY = 413
    DUTY_MANAGER = 1735

    ROBOT_MANAGER = 1260


class Idm:
    API_PATH = 'api/v1'
    PROD_URL = 'https://idm-api.yandex-team.ru/'  # Defaults _base_url

    # class State:
    #     # https://wiki.yandex-team.ru/Intranet/idm/states/
    #     GRANTED = "granted"
    #     REQUESTED = "requested,rerequested,review_request"
    #     DECLINED = "declined"
    #     EXPIRED = "expired"

    def __init__(
        self,
        token: str,
        base_url: str = PROD_URL
    ):
        self._base_url = Idm._canonize_url(base_url)
        self._api_url = os.path.join(self._base_url, Idm.API_PATH).strip('/')
        self._token = token.strip()
        self._base_headers = {
            "Content-Type": "application/json",
            "Authorization": f"OAuth {self._token}",
        }

    """================================================================================"""
    """================================ Public methods ================================"""
    """================================================================================"""

    @staticmethod
    def build_path(abc: Abc, service_id: int) -> Optional[str]:
        """
        Gets full slug path out of abc service id
        :param abc: Initialized instance of security.yaseclib.abc.Abc
        :param service_id: abc servic id
        :return: slug path to the service
        """
        service = abc.get_service_by_id(service_id, fields='path')
        path = service.get('path')
        if len(path) >= 990:  # There is a statement that field stores up to 1000 symbols (May hit undefined behaviour)
            logger.warning(f'Path {path} may be truncated!')
        return path

    @safe_func
    def get_rules_awaiting_approve(self, **kwargs) -> Optional[DictVal]:
        """
        GET {prefix}/approverequests/
        Возвращает: список подтверждений ролей, ожидающих либо ожидавших подтверждения от пользователя
        https://wiki.yandex-team.ru/intranet/idm/api/public/#spisokpodtverzhdenijj
        :param kwargs: status=pending|processed; approver, system, path, user, group
        :return:
        """
        if 'status' not in kwargs:
            kwargs.update({'status': 'pending'})
        res = DictVal(self.__get_complete('approverequests/', **kwargs))
        return res

    @safe_func
    def resolve_rule(
        self,
        rule_id: int,
        approve: bool,
        comment: str = None,
        _raw_return=False,
        **extra_params
    ) -> Optional[Union[bool, Response]]:
        """
        POST {prefix}/approverequests/<id>/
        Сохраняет решение подтверждающего
        https://wiki.yandex-team.ru/intranet/idm/api/public/#podtverzhdenielibootklonenie
        :param _raw_return: return Response instance
        :param rule_id: identifier of IDM rule
        :param approve: Boolean variable indicating verdict - True for Approve and False for Deny
        :param comment: Comment to leave
        :param extra_params: None yet
        :return: By default return True if succeeds
        """
        if not isinstance(approve, bool):
            logger.error('approve must be boolean value - True or False')
            return False
        res = self.__post(f'approverequests/{int(rule_id)}/', params=dict(
            approve=approve, comment=comment, **extra_params
        ))
        if _raw_return:
            return res
        return res.status_code in {200, 201, 202, 204}

    @safe_func
    def request_role(self, **params) -> Dict:
        # https://wiki.yandex-team.ru/intranet/idm/api/public/#zaprosyrolejj
        return self.__post('rolerequests', params).json()

    @safe_func
    def request_abc_role_for_user(
        self,
        user: str,
        path: str,
        role: RoleId,
        scope='*',
        silent=True,
        **params
    ) -> Dict:
        """
        https://wiki.yandex-team.ru/intranet/idm/api/public/#zaprosroli
        :param user: staff login
        :param path: slug path for service
        :param role: see RoleId
        :param scope: custom scope may be defined (defaults to '*')
        :param silent: whether notifications are required
        :param params: simulate, fields_data, comment, deprive_at, review_at, deprive_after_days
        :return:
        """
        return self.request_role(
            user=user,
            path=os.path.join(path, scope, str(role)),
            system='abc',
            silent=silent,
            **params
        )

    @safe_func
    def batch_request_roles(self, label: str, system: str, path: str, **extra) -> Dict:
        return self.__post('batchrolerequest', params=dict(
            label=label,
            system=system,
            path=path,
            **extra
        )).json()

    """================================================================================"""
    """============================= End of Public methods ============================"""
    """================================================================================"""

    @staticmethod
    def __get_path(path=None, params=None) -> str:
        return urlunparse((
            'https', 'idm-api.yandex-team.ru', os.path.join('api/v1', path), None, params, None
        ))

    @staticmethod
    def _canonize_url(url: str) -> str:
        return url.strip()

    def __get_complete(self, path, **params):
        url = os.path.join(self._api_url, path)
        r = get(url, params=params, headers=self._base_headers)
        json_data = r.json()
        rules = json_data.get("objects", [])
        meta = json_data["meta"]

        while meta.get("next") is not None:
            route = meta["next"]
            _, _, path, _, query, _ = urlparse(route)
            url = Idm.__get_path(path, query)
            r_next = get(url, headers=self._base_headers)
            json_data = r_next.json()
            transit = json_data.get("objects", [])
            rules = rules + transit
            meta = json_data.get("meta")
        return rules

    def __post(self, path=None, params=None):
        url = os.path.join(self._api_url, path)
        r = post(url, json=params, headers=self._base_headers)
        return r
