import requests
import json
import logging
import datetime

# try:
#     from urlparse import urljoin
# except ImportError:
#     from urllib.parse import urljoin

log = logging.getLogger(__name__)


class ImpulseQueuer(object):
    def __init__(self, base_url="https://impulse.sec.yandex-team.ru/api/v1", token=""):
        self.base_url = base_url
        self.token = token
        self.headers = {
            "Authorization": "OAuth " + self.token,
            "Content-type": "application/json",
        }

    def create_proj(self, name, org_id, slug="", tracker_queue=""):
        if not slug:
            slug = name
        data = {"name": name, "slug": slug, "tracker_queue": tracker_queue}

        create_proj_url = "%s%s%s%s" % (
            self.base_url,
            "/storage/organizations/",
            org_id,
            "/projects",
        )
        r = requests.post(
            url=create_proj_url,
            headers=self.headers,
            data=json.dumps(data),
            verify="/etc/ssl/certs/ca-certificates.crt",
        )

        result = r.json()
        if result["ok"]:
            return int(result["result"]["id"])
        else:
            return None

    def run_scan(self, org_id, proj_id, parameters, callback_url, workflow_id=None):
        run_scan_url = "%s%s" % (self.base_url, "/control/tasks")
        if workflow_id:
            data = {
                "organization_id": org_id,
                "project_id": proj_id,
                "workflow_id": workflow_id,
                "parameters": parameters,  # smt like
                "callback_url": callback_url,
            }
        else:
            if "python" not in parameters:  # bandit_scan parameter
                parameters["python"] = 3
            if "config_paths" not in parameters:  # semgrep_scan parameter
                parameters["config_paths"] = ["go", "java", "javascript", "python"]
            if "codeql_lang" not in parameters:  # codeql languages parameter
                parameters["codeql_lang"] = ",".join(
                    ["cpp", "go", "java", "python", "javascript"]
                )
            if "codeql_qls" not in parameters:  # codeql query suites parameter
                parameters["codeql_qls"] = ",".join(
                    [
                        "javascript-secaudit.qls",
                        "python-secaudit.qls",
                        "cpp-secaudit.qls",
                        "java-secaudit.qls",
                        "go-secaudit.qls",
                    ]
                )
            data = {
                "organization_id": org_id,
                "project_id": proj_id,
                "analysers": [
                    "yadi_scan",
                    "gosec_scan",
                    "yodax_scan",
                    "bandit_scan",
                    "semgrep_scan",
                    "codeql_scan",
                ],
                "parameters": parameters,  # smt like
                "callback_url": callback_url,
            }
        r = requests.post(
            url=run_scan_url,
            headers=self.headers,
            data=json.dumps(data),
            verify="/etc/ssl/certs/ca-certificates.crt",
        )

        result = r.json()
        if result["ok"]:
            return result["result"]["id"]
        else:
            return None

    def create_scan_cron(
        self, org_id, proj_id, parameters, analysers, cron_spec, description
    ):
        run_scan_url = "%s/control/organizations/%s/projects/%s/cron" % (
            self.base_url,
            org_id,
            proj_id,
        )
        data = {
            "organization_id": org_id,
            "project_id": proj_id,
            "analysers": analysers,
            "parameters": parameters,
            "spec": cron_spec,
            "description": description,
        }
        r = requests.post(
            url=run_scan_url,
            headers=self.headers,
            data=json.dumps(data),
            verify="/etc/ssl/certs/ca-certificates.crt",
        )
        result = r.json()
        if result["ok"]:
            return result["result"]["id"]
        else:
            return None

    def send_to_impulse(
        self, issue, org_id, parameters, callback_url, tracker_queue=""
    ):
        proj_id = self.create_proj(
            name=issue.key, org_id=org_id, tracker_queue=tracker_queue
        )
        if proj_id:
            self.run_scan(
                org_id=org_id,
                proj_id=proj_id,
                parameters=parameters,
                callback_url=callback_url.format(issue.key),
            )

    def get_org_projects(self, org_id):
        org_projects_url = "%s/storage/organizations/%s/projects" % (
            self.base_url,
            org_id,
        )
        r = requests.get(
            url=org_projects_url,
            headers=self.headers,
            verify="/etc/ssl/certs/ca-certificates.crt",
        )

        result = r.json()
        if result["ok"]:
            return result["result"]
        else:
            return []

    def get_proj_vulns(
        self, org_id, proj_id, date_isoformat=datetime.date.today().isoformat()
    ):
        previous_isoformat = (
            datetime.datetime.strptime(date_isoformat, "%Y-%m-%d").date()
            - datetime.timedelta(days=1)
        ).isoformat()
        offset = 0
        limit = 50
        more = True
        vulns_url = "%s/storage/organizations/%s" "/projects/%s/vulnerabilities" % (
            self.base_url,
            org_id,
            proj_id,
        )
        total = []
        while more:
            parameters = {
                "offset": offset,
                "limit": limit,
                "filter": '{"and":[{"gt":{"v.first_found_at":"%s"}},'
                '{"lt":{"v.first_found_at":"%s"}},'
                '{"eq":{"v.status":"not_reviewed"}}]}'
                % (previous_isoformat, date_isoformat),
            }
            r = requests.get(
                url=vulns_url,
                headers=self.headers,
                params=parameters,
                verify="/etc/ssl/certs/ca-certificates.crt",
            )

            result = r.json()
            if result["ok"]:
                if len(result["result"]) < limit:
                    more = False
                else:
                    offset += limit
                total.extend(result["result"])
            return total
