import tenacity
import requests
from urllib.parse import urljoin


class Moe(object):
    """
    Class for searching data from MOE API.
    """

    DEFAULT_MOE_TVM_ID = "2001279"

    def __init__(
        self, base_url="https://api.moe.yandex-team.ru", tvm_id="2001279", tvm_ticket=""
    ):
        self._base_url = base_url
        self._tvm_id = tvm_id
        self._tvm_ticket = tvm_ticket

    def get_users_result_by_course(
        self, course_id, percent=None, tvm_ticket=None, verify=False
    ):
        """
        Get users result by course with pagination. If you want
        pure response from MOE, don't use percent.
        Percent take first mark from all clessons>=value.
        Clessons with theory are always 0.
        Clessons mark range is from 0 to 100.

        :param course_id: Course id for search
        :type course_id: int
        :param tvm_ticket: Ticket for auth
        :type tvm_ticket: str
        :param percent: Minimum percent to сourse completion.
        :type percent: float
        :param verify: SSL config
        :type verify: bool or path to cert
        :return: If you use percent - logins with marks,
        list with pure response by paginated page
        :rtype: list
        """

        tvm_ticket_ = tvm_ticket or self._tvm_ticket
        result = []
        path = "/api/v3/courses/{}/students_results/?page_size=150&page={}"
        page = 1
        while page:
            rsp = self._get_request(path.format(course_id, page), tvm_ticket_, verify)
            if percent:
                for item in rsp["results"]:
                    mark = list(map(lambda x: x["result"]["percent"], item["clessons"]))
                    result.append(
                        (item["login"], self._get_lessons_result(mark, percent))
                    )
            else:
                result.append(rsp)
            page = rsp.get("next")
        return result

    def get_user_result_by_course(
        self, course_id, login, percent=None, tvm_ticket=None, verify=False
    ):
        """
        Get user result by course. If you want pure response from
        MOE, don't use percent.
        Percent take first mark from all clessons>=value. Clessons
        with theory are always 0.
        Clessons mark range is from 0 to 100.

        :param course_id: Course id for search
        :type course_id: int
        :param login: Login for search
        :type login: str
        :param percent: Minimum percent to сourse completion.
        :type percent: float
        :param tvm_ticket: Ticket for auth
        :type tvm_ticket: str
        :param verify: SSL config
        :type verify: bool or path to cert
        :return: If you use percent - user with mark, list with pure
        response
        :rtype: list
        """

        tvm_ticket_ = tvm_ticket or self._tvm_ticket
        result = []
        path = "/api/v3/courses/{}/students_results/?login={}&page={}"
        page = 1
        while page:
            rsp = self._get_request(
                path.format(course_id, login, page), tvm_ticket_, verify
            )
            if percent:
                for item in rsp["results"]:
                    mark = list(map(lambda x: x["result"]["percent"], item["clessons"]))
                    result.append(
                        (item["login"], self._get_lessons_result(mark, percent))
                    )
            else:
                result.append(rsp)
            page = rsp.get("next")
        return result

    @tenacity.retry(
        retry=tenacity.retry_if_exception_type(requests.exceptions.RequestException),
        wait=tenacity.wait_random(min=0.5, max=2),
        stop=tenacity.stop_after_attempt(5),
        reraise=True,
    )
    def _get_request(self, url, ticket, verify):
        url = urljoin(self._base_url, url)
        rsp = requests.get(
            url, headers={"X-Ya-Service-Ticket": "{}".format(ticket)}, verify=verify
        )
        return rsp.json()

    @staticmethod
    def _get_lessons_result(mark, percent):
        """
        Take users clessons mark, compare with percent and return
        course completion status.
        Return True for first complete clessons because clessons with
        theory are always 0.

        :param mark: User's clessons mark
        :type mark: int
        :param percent: Minimum percent to сourse completion.
        :type percent: float
        :return: Course completion status
        :rtype: bool
        """
        for _ in mark:
            grade_scorm = float(_) if _ is not None else 0.0
            if grade_scorm >= percent:
                return True
        return False
