import collections
import logging
import operator
import re
from calendar import timegm
from datetime import datetime
from typing import Optional, List

import requests
import tenacity
from cachetools import cachedmethod, TTLCache

# from typing import Optional, Union  # Doesn't exist for py2

logger = logging.getLogger(__name__)


class Staff(object):
    base_url = ""
    token = ""
    retries_delay_random_min = 0.5
    retries_delay_random_max = 2
    retries_count = 3

    def __init__(self, base_url, token):
        self.base_url = base_url.strip()
        self.token = token.strip()
        self.base_headers = {
            "content-type": "application/json",
            "Authorization": "OAuth " + self.token,
        }

        self.persons_info_cache = TTLCache(maxsize=100, ttl=600)
        self.group_members_cache = TTLCache(maxsize=10, ttl=600)

    def get_persons(self, group_name=None, join_at=None, limit=None):
        payload = {
            "groups.group.type": "servicerole",
            "official.is_dismissed": "false",
            "official.is_robot": "false",
        }

        if join_at:
            payload["official.join_at"] = '"' + join_at + '"'

        if limit:
            payload["_limit"] = str(limit)

        if group_name:
            payload["_query"] = 'groups.group.name==regex("' + group_name + '.*")'

        url = self.base_url + "/persons"
        r = requests.get(url, params=payload, headers=self.base_headers)
        json_data = r.json()
        return json_data.get("result", [])

    def get_persons_by_depart_url(self, department_url, join_at=None, limit=None):
        payload = {"official.is_dismissed": "false", "official.is_robot": "false"}

        if join_at:
            payload["official.join_at"] = '"' + join_at + '"'

        if limit:
            payload["_limit"] = str(limit)

        payload["_query"] = (
            'department_group.department.url==regex("' + department_url + '.*")'
        )
        url = self.base_url + "/persons"
        r = requests.get(url, params=payload, headers=self.base_headers)
        json_data = r.json()
        return json_data.get("result", [])

    def get_person_departments(self, login):
        # type: (str) -> str
        """
        Returns department groups of login
        :param login:
        :type login str
        :return: List[Dict]
        """
        url = self.base_url + '/persons'
        payload = {
            '_query': f'login=="{login}"',
            '_one': 1,
            '_fields': 'department_group'
        }
        r = requests.get(url, params=payload, headers=self.base_headers)
        json_data = r.json()
        d_group = json_data.get('department_group')
        if d_group:
            ancestors = d_group.get('ancestors')
            department = d_group.get('department')
            return ancestors + [department]

    def is_yandexoid(self, login):
        # type: (str) -> bool
        """
        Check that provided login:
        1. exist on staff
        2. not dismissed
        3. not robot

        :param login: Login for check
        :type login: str
        :return: True, then user with such login exist on Staff, otherwise -
            False.
        :rtype: bool
        """

        try:
            if self.is_robot(login) or self.is_dismissed(login):
                result = False
            else:
                result = True
        except ValueError:
            result = False

        return result

    def is_dismissed(self, login):
        # type: (str) -> bool
        """
        Check employee status by login.
        If login does not exist - exception raised.

        :param login: User login
        :type login: str
        :return: True if user is dismissed, otherwise - False
        :rtype: bool
        """
        return self.get_person_info(login=login, field="official.is_dismissed")

    def is_homeworker(self, login):
        # type: (str) -> bool
        """
        Check, that provided user login is homeworker.
        If login does not exist - exception raised.

        :param login: User login
        :type login: str
        :return: True if user is homeworker, otherwise - False
        :rtype: bool
        """
        return self.get_person_info(login=login, field="official.is_homeworker")

    def is_external(self, login):
        # type: (str) -> bool
        """
        Check, that provided user login is external.
        If login does not exist - exception raised.

        .. note::
            User will be external, if:
            - he is robot
            - he is in External consultants department (
                group.url="ext")

        :param login: User login
        :type login: str
        :return: True if user is external, otherwise - False
        :rtype: bool
        """
        affiliation = self.get_person_info(login=login, field="official.affiliation")

        result = True
        if not self.is_homeworker(login):
            if affiliation != "external":
                result = False
        else:
            if not self.check_user_in_department(login=login, group_url="ext"):
                return False

        return result

    def get_person_info(self, login, field):
        # type (str, str) -> Union[int, str, list, dict, None]
        """
        Get value of required field from user Staff page.
        Field can be dot notated, to get nested JSON values.

        :param login: User login
        :type login: str
        :param field: API field to be queried
        :type field: str
        :return: field value
        :rtype: int, str, list, dict, None
        :raises: ValueError - if login is None or empty string
        """

        if login is None or not login:
            raise ValueError("Empty login is not allowed")

        fields_splitted = field.split(".")

        value = None
        for field_level in fields_splitted:
            if value is None:
                full_person_info = self.get_full_person_info(login)
                value = full_person_info[field_level]
            else:
                if (
                    isinstance(value, list) and
                    len(value) > 0 and
                    field_level in value[0]
                ):
                    value = value[0][field_level]
                elif isinstance(value, dict) and field_level in value:
                    value = value[field_level]
                else:
                    value = None

        return value

    @tenacity.retry(
        retry=tenacity.retry_if_exception_type(requests.exceptions.RequestException),
        wait=tenacity.wait_random(
            min=retries_delay_random_min, max=retries_delay_random_max
        ),
        stop=tenacity.stop_after_attempt(retries_count),
        reraise=True,
    )
    @cachedmethod(operator.attrgetter("persons_info_cache"))
    def get_full_person_info(self, login):
        # type: (str) -> dict
        """
        Query user info using /persons api call with _pretty=1.
        With retries controlled by class attributes.

        .. note::
            1. Results are cached.
            2. Retrying only on requests lib exceptions

        :param login: User login
        :type login: str
        :return: Json answer from API
        :rtype: dict
        :raises: ValueError - if login not found on Staff
        :raises: ValueError - if login is None or empty string
        """

        if login is None or not login:
            raise ValueError("Empty login is not allowed")

        url = "/".join([self.base_url, "persons"])

        payload = {"_pretty": 1, "login": login, "_one": 1}

        result = self._requests_get(url=url, params=payload, headers=self.base_headers)

        if "error_message" in result:
            raise ValueError(
                "API returned error for login='{}'. "
                "Error message: {}".format(login, result["error_message"])
            )

        return result

    def check_user_exist_on_staff(self, login):
        # type: (str) -> bool
        """
        More simple request to Staff API, if you need to check
            only user existence on Staff.
        Use get_full_person_info, if you will after fetch user info from Staff.

        .. note::
            Does not check user groups or employee status (dismissed or not).

        :param login: User login
        :type login: str
        :return: True if exist, otherwise false.
        :rtype: bool
        """

        url = "/".join([self.base_url, "persons"])

        params = {"login": login, "_fields": "login"}

        response = self._requests_get(url=url, headers=self.base_headers, params=params)

        if response["result"]:
            result = True
        else:
            result = False

        return result

    def get_group_info(self, group_url, fields=None):
        # type: (str, Optional[list]) -> dict
        """
        Query user info using /persons api call with _pretty=1.

        :param group_url: Staff group url (codename)
        :type group_url: str
        :param fields: Fields to query. Optional, defaults to *None*
        :type fields: list
        :return: Json answer from API
        :rtype: dict
        """
        url = "/".join([self.base_url, "groups"])

        payload = {"_pretty": 1, "_one": 1, "department.url": group_url}

        if isinstance(fields, list) and fields:
            payload["_fields"] = ",".join(fields)

        try:
            request = requests.get(url, params=payload, headers=self.base_headers)
            result = request.json()
        except Exception as exc:
            logger.error("Request failed! Raised exception: {}".format(exc))
            result = {}

        return result

    def get_group_info_by_url(self, entity_slug, fields=None):
        # type: (str, Optional[list]) -> dict
        """
        Query group info by url= query param.

        :param entity_slug: entity slug (Staff department or ABC-role)
        :type entity_slug: str
        :param fields: Fields to query. Optional, defaults to *None*
        :type fields: list
        :return: Json answer from API
        :rtype: dict
        """
        url = "/".join([self.base_url, "groups"])

        payload = {"_pretty": 1, "_one": 1, "url": entity_slug}

        if isinstance(fields, list) and fields:
            payload["_fields"] = ",".join(fields)

        try:
            request = requests.get(url, params=payload, headers=self.base_headers)
            result = request.json()
        except Exception as exc:
            logger.error("Request failed! Raised exception: {}".format(exc))
            result = {}

        return result

    def get_person_department_chief(self, login):
        # type: (str) -> Optional[str]
        """
        Get user department head login

        :param login: User login
        :type login: str
        :return: Departement head login
        :rtype: str
        """
        boss_login = None
        boss_logins = self.get_person_info(
            login=login, field="department_group.parent.department.heads"
        )
        for login in boss_logins:
            if login.get("role") == "chief":
                boss_login = login.get("person").get("login")

        return boss_login

    def get_person_boss(self, login):
        # type: (str) -> str
        """
        Get user boss, via :func:`get_person_info()`.

        :param login: User login for which to query boss login
        :type login: str
        :return: Boss login
        :rtype: str
        :raises: ValueError - if login is None or empty string
        """

        if login is None or not login:
            raise ValueError("Empty login is not allowed")

        boss_login = None

        boss_logins = self.get_person_info(
            login=login, field="department_group.department.heads"
        )

        for boss in boss_logins:
            if boss.get("role") == "chief":
                boss_candidate = boss.get("person").get("login")
                if boss_candidate != login:
                    boss_login = boss_candidate
                break

        if boss_login is None:
            try:
                boss_login = self.get_person_department_chief(login)
            except Exception:
                return ""

        return boss_login

    def get_robot_responsible(self, login):
        # type: (str) -> Optional[str]
        """
        Get robot responsible if it exist on Staff

        :param login: Robot login
        :type login: str
        :return: Responsible user login
        :rtype: str
        """

        try:
            # If robot has no owners, then field robot_owners does not exist
            response = self.get_person_info(login=login, field="robot_owners")

            responsible_login = response[0]["person"]["login"]
        except (IndexError, KeyError):
            responsible_login = None

        return responsible_login

    def is_robot(self, login):
        # type: (str) -> bool
        """
        Check, if provided login is robot or zombie

        :param login: Login for check
        :type login: str
        :return: True if is, otherwise - False
        :rtype: bool
        :raises: ValueError - if provided login not found on Staff
        """

        return self.get_person_info(login=login, field="official.is_robot")

    def check_user_in_department(self, login, group_url):
        # type: (str, str) -> bool
        """
        Check if user is in department by group.url value.
        Example: yandex_mnt_security

        :param login: User login
        :type login: str
        :param group_url: group.url API value
        :type group_url: str
        :return: True if user in department
        :rtype: bool
        """
        url = "https://staff-api.yandex-team.ru/v3/groupmembership"

        params = {
            "person.login": login,
            "_fields": "group.url,group.ancestors.department.url",
            "_query": '(group.url=="{0}" or '
                      'group.ancestors.department.url=="{0}")'.format(group_url),
        }

        result_json = self._requests_get(
            url=url, headers=self.base_headers, params=params
        )

        if result_json["result"]:
            result = True
        else:
            result = False

        return result

    def is_person_from_security_group(self, login):
        # type: (str) -> bool
        """
        Check, if user is working in Service of informational security group.
        Via :func:`get_person_info()`.

        :param login: User login
        :type login: str
        :return: True if yes, otherwise False
        :rtype: bool
        """

        return self.check_user_in_department(
            login=login, group_url="yandex_mnt_security"
        )

    def get_person_office_name(self, login):
        # type: (str) -> Optional[str]
        """
        Query user office name via :func:`get_person_info()`.

        :param login: User login
        :type login: str
        :return: Office name or *None* (if office info is absent)
        :rtype: str
        """

        return self.get_person_info(login, field="location.office.name.en")

    def get_person_chief_list(self, login: str) -> List:
        # type: (str) -> list
        """
        Get user boss, via :func:`get_person_info()`.
        The boss order is descending, down from volozh.

        :param login: User login for which to query bosses logins list
        :type login: str
        :return: list of bosses logins
        :rtype: list
        """

        response = self.get_person_info(login=login, field="groups.group.ancestors")
        heads_logins = []

        for department in response:
            heads = department["department"].get("heads")
            if heads:
                for head in heads:
                    if head["role"] == "chief":
                        heads_logins.append(head["person"]["login"])
        boss = self.get_person_boss(login)
        if boss:
            heads_logins.append(boss)

        return list(collections.OrderedDict.fromkeys(heads_logins))

    def _clean_empty(self, d):
        if not isinstance(d, (dict, list)):
            return d
        if isinstance(d, list):
            return [v for v in (self._clean_empty(v) for v in d) if v]
        return {k: v for k, v in ((k, self._clean_empty(v)) for k, v in d.items()) if v}

    def get_person_group(self, user, url=False, eng_name=True):
        # type: (str, bool, bool) -> str
        """
        Return person group membership.


        :param user: person login
        :type user: str or str
        :param url: Return group url, not name (like yandex_mnt_fin_service).
         Defaults to *False*
        :type url: bool
        :param eng_name: Return group name on English,
         otherwise on Russian Defaults to *True*
        :type eng_name: bool
        :return: Person group name or url
        :rtype: str or str
        """

        membership_url = "https://staff-api.yandex-team.ru/v3/groupmembership"
        params = {"_pretty": 1, "person.login": user}
        if url:
            params["_fields"] = "group.department.url"

        elif eng_name:
            params["_fields"] = "group.department.name.full.en"
        elif not eng_name:
            params["_fields"] = "group.department.name.full.ru"

        try:
            request = requests.get(
                membership_url, params=params, headers=self.base_headers
            )
            result = request.json()
        except Exception as exc:
            logger.error("Request failed! Raised exception: {}".format(exc))
            result = {}

        result = self._clean_empty(result.get("result"))[0]

        keys = params["_fields"].split(".")

        for key in keys:
            r = result.get(key)
            if r is not None:
                result = r
            else:
                break

        return result

    def get_last_logon(self, login):
        # type: (str) -> Optional[int]
        """
        Get user last logon time.

        :param login: User login for which to get logon time
        :type login: str
        :return: User logon time in epoch format
        :rtype: int (None will be returned if server is not responding
         or response can't be parsed)
        """
        url = (
            "https://center.yandex-team.ru/api/v1/user/"
            "{}/where.json?fields=updated_at".format(login)
        )
        try:
            res = requests.get(url, headers=self.base_headers)
            if res.status_code != 200:
                return None
            json_data = res.json()
        except Exception as exc:
            logger.error("Request failed! Raised exception: {}".format(exc))
            json_data = []

        if not json_data or "updated_at" not in json_data[0]:
            return None
        updated_at = json_data[0]["updated_at"]
        timestamp = datetime.strptime(updated_at, "%Y-%m-%dT%H:%M:%S")
        epoch = int(timegm(timestamp.utctimetuple()))

        return epoch

    @staticmethod
    def _requests_get(url, headers, params=None):
        # type: (str, dict, dict) -> dict
        """
        Wrapper for requests.get with response checking and returning
            results as dict (via json() method).

        :param url: Url to use for requesst
        :type url: str
        :param headers: Request headers
        :type headers: dict
        :param params: Request params
        :type params: dict
        :return: Result in json
        :rtype: dict
        :raises: RuntimeError - if bad response status code
        """
        response = requests.get(url=url, headers=headers, params=params)
        result = response.json()

        if response.status_code != 200 or result is None:
            logger.error(
                "Rsponse.json is None or bad response "
                "status code: %s. "
                "Response json: %s",
                response.status_code,
                result,
            )
            raise RuntimeError("Bad response status code!")

        return result

    @cachedmethod(operator.attrgetter("group_members_cache"))
    def get_group_members(
        self,
        group_url,  # type: str
        group_type=None,  # type: Optional[str]
        show_dismissed=False,  # type: bool
        with_subgroups=True,  # type: bool
        limit=100,  # type: int
        max_results_count=None,  # type: Optional[int]
    ):
        # type: (...) -> list
        """
        Return staff group members logins.

        :param group_url: Staff group url (codename)
        :type group_url: str
        :param group_type: For different group types ABC, Wiki groups.
            Default is Staff group.
        :type group_type:
        :param show_dismissed: Show dissmissed group members or not.
         Defaults to *False*
        :type show_dismissed: bool
        :param with_subgroups: Include subgroups members of provided group.
         Defaults to *True*
        :type with_subgroups: bool
        :param limit: Limit per single page (to avoid pagination of
            default 50 value)
        :type limit: int
        :param max_results_count: Max number of results allowed
            (do not follow next if not required). Default is None.
        :type max_results_count: int
        :return: List with group members login
        :rtype: list
        """

        url = "https://staff-api.yandex-team.ru/v3/groupmembership"

        params = {
            "_limit": limit,
            "person.official.is_dismissed": show_dismissed,
            "_fields": "person.login",
            "group.type": group_type,
        }

        if with_subgroups:
            params.update(
                {
                    "_query": "group.url=="
                              'regex("{group_url}.*")'.format(group_url=group_url),
                }
            )
        else:
            params.update(
                {
                    "group.url": group_url,
                }
            )

        result = self._requests_get(url=url, params=params, headers=self.base_headers)

        logins = {entry["person"]["login"] for entry in result["result"]}

        while result.get("links") and result["links"].get("next") is not None:
            if max_results_count is not None and len(logins) >= max_results_count:
                break

            url = result["links"]["next"]
            result = self._requests_get(url=url, headers=self.base_headers)
            logins.update([entry["person"]["login"] for entry in result["result"]])

        return list(logins)

    def get_group_members_phone_numbers(
        self,
        group_url,
        show_dismissed=False,
        with_subgroups=True,
        limit=100,
        max_results_count=None,
        members=None,
    ):
        # type: (str, bool, bool, int, Optional[int], Optional[list]) -> list
        """
        Get group members primary phone numbers.

        :param group_url: Group url name (for example: yandex_mnt_security)
        :type group_url: str
        :param show_dismissed: Default to False. If True,
            dismissed employees will be returned as well.
        :type show_dismissed: bool
        :param with_subgroups: Defaults to True: all child groups members
            will be returned.
        :type with_subgroups: bool
        :param limit: How much rsults to return per single request (
            for pagination)
        :type limit: int
        :param max_results_count: Max number of results allowed
            (do not follow next if not required). Default is None.
        :type max_results_count: int
        :param members: You can provide logins as a list, to get their
            phone numbers.
            Usage example:
            get_group_members_phone_numbers(group_url="", members=["lx"])
        :type members: list
        :return: List with primary phone numbers of all gorup members
        :rtype: list
        """
        if members is None:
            logins = self.get_group_members(
                group_url=group_url,
                show_dismissed=show_dismissed,
                with_subgroups=with_subgroups,
                limit=limit,
                max_results_count=max_results_count,
            )
        else:
            logins = members

        numbers = [
            self.get_person_info(login=login, field="phones.number") for login in logins
        ]

        numbers = [re.sub(r"[\s()-]", "", number) for number in numbers if number is not None]

        return numbers

    def get_all_robots(self, limit=10000, is_dismissed=False):
        url = "/".join([self.base_url, "persons"])
        payload = {
            "official.is_dismissed": is_dismissed,
            "official.is_robot": True,
            "_fields": "login,official.join_at,official.quit_at,robot_owners",
            "_limit": limit,
        }
        response = self._requests_get(
            url=url, params=payload, headers=self.base_headers
        )
        result = response["result"]

        while response.get("links") and response["links"].get("next") is not None:
            url = result["links"]["next"]
            response = self._requests_get(url=url, headers=self.base_headers)
            result.append(response.get("result", []))

        return result

    def get_login_by_staff_id(self, id):

        url = "/".join([self.base_url, "persons"])
        payload = {"id": id, "_fields": "login"}
        response = self._requests_get(
            url=url, params=payload, headers=self.base_headers
        )
        result = response["result"]["login"]

        return result

    def get_department_members_by_level(
        self,
        level: int,
        url: Optional[str] = None,
        results: Optional[list] = None,
        filters: Optional[dict] = None,
    ) -> list:
        """
        Returns all members of groups by department level.
        Department level 0 - Yandex departemnt (group.url=yandex).

        You can pass any additional API arguments via filters argument.

        More info in SOC-1170.

        :param level: Max department level (if 3, will search for department_group.level=0,1,2)
        :type level: int
        :param url: Next url for pagination. If None - then fetch first page
        :type url: str
        :param results: User logins
        :type results: list
        :param filters: Additional arguments to pass
        :type filters: dict
        :return: User logins list
        :rtype: list
        """

        payload = {
            "department_group.level": ",".join([str(i) for i in range(level)]),
            "_fields": "login",
        }

        if results is None:
            results = list()

        if filters is not None:
            payload.update(filters)

        if url is None:
            url = "/".join([self.base_url, "persons"])

        response = self._requests_get(
            url=url, params=payload, headers=self.base_headers
        )

        result = response["result"]

        for user in result:
            results.append(user["login"])

        if "links" in response and "next" in response["links"]:
            url = response["links"]["next"]
            self.get_department_members_by_level(level, url, results, filters)

        return results
