import requests
import dns.resolver
import time


class Waffles(object):
    TVM_ID = "2008801"

    def __init__(
        self,
        base_url="https://waffles.sec.yandex.net/api/v1/virtual_host/",
        tvm_ticket="",
    ):
        self._base_url = base_url
        self._tvm_ticket = tvm_ticket

    def get_virtual_hosts(self, ip, tvm_ticket=None, verify=True):
        """
        Request Waffles API to get virtual hosts for IP address.

        ip:
            IPv4 or IPv6.
        tvm_ticket:
            Optional. if not set on constructor.
            Waffles requires tvm ticket to use its API.
        verify:
            Optional.
            Verify with "/etc/ssl/certs/ca-certificates.crt"
            or not verify at all.

        return:
            List of vhosts OR empty list
        """

        tvm_ticket_ = tvm_ticket or self._tvm_ticket
        url = self._base_url + ip
        verify_ = "/etc/ssl/certs/ca-certificates.crt" if verify else False

        # XXX: this is kostyli
        for retry in range(2):
            r = requests.get(
                url, headers={"X-Ya-Service-Ticket": tvm_ticket_}, verify=verify_
            )
            if r.status_code != 403:
                break
            time.sleep(10)

        try:
            vhosts = r.json()
        except Exception:
            return []

        if not vhosts or not isinstance(vhosts, dict):
            return []

        return list() if vhosts.get(ip) is None else vhosts.get(ip)

    def do_resolve(self, hostname, depth=0):
        if depth > 5:
            return None
        dns_ans = None
        # try ipv6 first
        try:
            dns_ans = dns.resolver.query(hostname, "AAAA")
        except dns.resolver.NoAnswer:
            pass
        except Exception:
            return None
        if not dns_ans:
            try:
                dns_ans = dns.resolver.query(hostname, "A")
            except dns.resolver.NoAnswer:
                return None
            except Exception:
                return None
        CNAME = 5
        for rec in dns_ans.response.answer:
            if rec.rdtype == CNAME:
                return self.do_resolve(str(rec).split()[-1].strip("."), depth + 1)
            return str(rec).split()[-1].strip(".")
