import os

import json
from flask import Flask, render_template, request, jsonify
import logging
from io import StringIO

import gixy
from gixy.core.manager import Manager
from gixy.core.config import Config
from gixy.core.exceptions import InvalidConfiguration
from yodax.formatters.json import JsonFormatter

from log_handler import LogHandler

app = Flask(__name__)
logging.captureWarnings(True)

DEFAULTS = {
    'origins': {
        'domains': ['yandex.ru',
                    'yandex.by',
                    'yandex.com',
                    'yandex.com.tr',
                    'yandex.kz',
                    'yandex.ua',
                    'yandex.fr',
                    'yandex.az',
                    'yandex.com.am',
                    'yandex.com.ge',
                    'yandex.co.il',
                    'yandex.kg',
                    'yandex.lt',
                    'yandex.lv',
                    'yandex.md',
                    'yandex.tj',
                    'yandex.tm',
                    'yandex.uz',
                    'yandex.ee',
                    'webvisor.com',
                    'ya.ru',
                    'yandex-team.ru'],
        'https_only': True
    }
}


@app.template_filter('autoversion')
def autoversion_filter(filename):
    fullpath = os.path.join(os.path.realpath(__file__), '..', filename[1:])
    fullpath = os.path.realpath(fullpath)
    try:
        timestamp = str(os.path.getmtime(fullpath))
    except OSError:
        return filename
    newfilename = "{0}?v={1}".format(filename, timestamp)
    return newfilename


@app.after_request
def add_security_headers(response):
    response.headers['X-Frame-Options'] = 'SAMEORIGIN'
    response.headers[
        'Content-Security-Policy'] = "default-src 'none'; script-src 'self' 'unsafe-eval' yastatic.net; style-src 'self' yastatic.net; img-src 'self'; font-src yastatic.net; connect-src 'self';"
    return response


def check_config(config, filename='<request>'):
    logger = logging.getLogger()
    logger.setLevel(logging.WARNING)
    log_handler = LogHandler()
    logger.addHandler(log_handler)

    yodax_config = Config(allow_includes=False)
    for plug, opts in DEFAULTS.items():
        yodax_config.set_for(plug, opts)

    formatter = JsonFormatter()
    result = {}
    try:
        with Manager(config=yodax_config) as yoda:
            yoda.audit(filename, StringIO(config), is_stdin=True)
            formatter.feed(filename, yoda)

        reports = json.loads(formatter.flush())
        warnings = log_handler.flush()
        warnings = [{'name': x['name'], 'message': x['msg'] % x['args']} for x in warnings]
        result = {'status': 'ok', 'result': reports, 'warnings': warnings}
    except InvalidConfiguration as e:
        result = {'status': 'failed', 'error': 'Can\'t parse your config \uD83D\uDE13: ' + str(e)}
    except Exception as e:
        result = {'status': 'failed', 'error': 'Error occurred during analyze: ' + str(e)}
    finally:
        logger.removeHandler(log_handler)
        log_handler.close()
    return result


@app.route('/validate', methods=['POST'])
def validate():
    if request.mimetype == 'application/json':
        config = request.get_json()['config']
    else:
        return jsonify({'status': 'failed', 'error': 'Unsupported content-type'})

    return jsonify(check_config(config))


@app.route('/api/check', methods=['POST'])
def api_check():
    if request.mimetype == 'application/nginx':
        config = request.get_data().decode('utf-8')
    elif request.mimetype == 'application/json':
        config = request.get_json()['config']
    else:
        return jsonify({'status': 'failed', 'error': 'Unsupported content-type'})

    return jsonify(check_config(config))


@app.route('/api/bundle', methods=['POST'])
def api_bundle():
    if request.mimetype == 'application/json':
        configs = request.get_json()
    else:
        return jsonify({'status': 'failed', 'error': 'Unsupported content-type'})

    results = {'results': {}, 'skips': {}}
    for meta in configs['files']:
        filename = meta['file_name']
        config = meta['content']
        yresult = check_config(config, filename=filename)
        if yresult['status'] == 'failed':
            results['skips'][filename] = yresult['error']
        elif yresult['result']:
            results['results'][filename] = yresult['result']

    return jsonify(results)


@app.route('/ping')
def ping():
    return 'OK'


@app.route('/')
def index():
    return render_template('index.html', config=request.args.get('c', ''), version=gixy.version)


if __name__ == '__main__':
    app.run()
