#include "extractor.h"
#include <ydb/library/yql/sql/sql.h>
#include <ydb/library/yql/ast/yql_type_string.h>

namespace NDeclMiner {
    namespace {
        const ui16 kSyntaxVersion = 1;

        using TTokenIterator = NSQLTranslation::TParsedTokenList::const_iterator;

        struct TDeclare {
            TString Name;
            TString Type;
        };

        class TDeclareParser {
        public:
            TDeclareParser(TTokenIterator curr, TTokenIterator end) : curr(curr), end(end) {}

            TResult<TDeclare> Parse() {
                auto variableName = parseVariableName();
                if (!variableName) {
                    return *lastErr;
                }

                skipWs();
                if (!chopToken("AS")) {
                    return *lastErr;
                }

                auto variableType = parseTypeString();
                if (!variableType) {
                    return *lastErr;
                }

                return TDeclare{
                        .Name = *variableName,
                        .Type = *variableType,
                };
            }

            TTokenIterator CurPos() {
                return curr;
            }

        private:
            TMaybe<TString> parseVariableName() {
                skipWs();

                if (!chopToken("DOLLAR")) {
                    lastErr = TError{} << "unexpected  DOLLAR token, but got: " << curr->Name;
                    return Nothing();
                }

                if (curr == end) {
                    lastErr = TError{} << "unexpected EOF";
                    return Nothing();
                }

                if (curr->Name != "ID_PLAIN") {
                    lastErr = TError{} << "unexpected  ID_PLAIN token, but got: " << curr->Name;
                    return Nothing();
                }

                return "$" + (curr++)->Content;
            }

            TMaybe<TString> parseTypeString() {
                skipWs();

                TStringStream out;
                while (curr != end) {
                    if (curr->Name == "COMMENT") {
                        continue;
                    }

                    if (curr->Name == "SEMICOLON") {
                        return out.Str();
                    }

                    out.Write(curr->Content);
                    ++curr;
                }

                lastErr = TError{} << "unexpected EOF";
                return Nothing();
            };

            void skipWs() {
                while (curr != end) {
                    if (curr->Name != "WS" && curr->Name != "COMMENT") {
                        break;
                    }

                    ++curr;
                }
            }

            bool chopToken(const TString &tokenName) {
                if (curr == end) {
                    lastErr = TError{} << "unexpected EOF";
                    return false;
                }

                if (curr->Name != tokenName) {
                    lastErr = TError{} << "expected " << tokenName << " token, but got: " << curr->Name;
                    return false;
                }

                ++curr;
                return true;
            }

            TMaybe<TError> lastErr;
            TTokenIterator curr;
            TTokenIterator end;
        };

        bool hasYqlErrors(const NYql::TIssues &issues) {
            return AnyOf(issues, [](const auto &issue) {
                return issue.GetSeverity() == NYql::TSeverityIds::S_ERROR;
            });
        }

        TResult<NSQLTranslation::TParsedTokenList> parseQuery(const TString &query, const TString &queryName) {
            NSQLTranslation::TTranslationSettings settings;
            google::protobuf::Arena arena;
            settings.Arena = &arena;
            settings.SyntaxVersion = kSyntaxVersion;
            settings.AnsiLexer = false;
            settings.WarnOnV0 = false;
            settings.V0Behavior = NSQLTranslation::EV0Behavior::Disable;

            NYql::TIssues issues;
            NSQLTranslation::TParsedTokenList tokens;
            auto onNextToken = [&tokens](NSQLTranslation::TParsedToken&& token) {
                tokens.push_back(std::move(token));
            };

            auto lexer = NSQLTranslation::SqlLexer(query, issues, settings);
            lexer->Tokenize(query, queryName, onNextToken, issues, NSQLTranslation::SQL_MAX_PARSER_ERRORS);
            if (hasYqlErrors(issues)) {
                return TError{} << "unable to parse query: " << issues.ToOneLineString();
            }

            return std::move(tokens);
        }
    }

    THolder<NDeclMiner::TQuery> TQuery::Parse(const TString &query, const TString &queryName) {
        auto maybeTokens = parseQuery(query, queryName);
        if (auto e = GetError(maybeTokens)) {
            ythrow yexception() << e->AsStrBuf();
        }

        return THolder<TQuery>(new TQuery{query, std::move(std::get<0>(maybeTokens))});
    }

    THashMap<TString, TDataType> TQuery::ResolveDeclares() {
        THashMap<TString, TDataType> out;

        TMemoryPool pool(4096);
        NYql::TIssues issues;

        auto tokenIt = tokens.cbegin();
        while (tokenIt != tokens.cend()) {
            if (tokenIt->Name != "DECLARE") {
                ++tokenIt;
                continue;
            }

            ++tokenIt;
            auto parser = TDeclareParser(tokenIt, tokens.cend());
            const auto maybeDecl = parser.Parse();
            if (auto e = GetError(maybeDecl)) {
                ythrow yexception() << "unable to parse declare at line " << tokenIt->Line << ": " << e;
            }

            const auto &decl = std::get<0>(maybeDecl);
            auto node = NYql::ParseType(decl.Type, pool, issues);
            if (!node) {
                ythrow yexception() << "unable to parse variable " << decl.Name << " (line:" << tokenIt->Line << "): " << issues.ToOneLineString();
            }

            const auto &maybeDataType = ParseDataType(node);
            if (auto e = GetError(maybeDataType)) {
                ythrow yexception() << "unexpected variable " << decl.Name << " (line:" << tokenIt->Line << "): type declaration: " << e;
            }

            out[decl.Name] = std::get<0>(maybeDataType);

            tokenIt = parser.CurPos();
        }

        return out;
    }

}
