#include "type_parser.h"
#include <util/string/cast.h>

namespace NDeclMiner {
    namespace {
        using typeMaker = std::function<TResult<TDataType>(const NYql::TAstNode *)>;

        EKind parseTypeKind(const NYql::TAstNode *node) {
            static const THashMap<TStringBuf, EKind> map = {
                    {TStringBuf("DataType"),      KIND_PRIMITIVE},
                    {TStringBuf("OptionalType"),  KIND_OPTIONAL},
                    {TStringBuf("ListType"),      KIND_LIST},
                    {TStringBuf("DictType"),      KIND_DICT},
                    {TStringBuf("TupleType"),     KIND_TUPLE},
                    {TStringBuf("StructType"),    KIND_STRUCT},
                    {TStringBuf("VariantType"),   KIND_VARIANT},
                    {TStringBuf("VoidType"),      KIND_VOID},
                    {TStringBuf("NullType"),      KIND_NULL},
                    {TStringBuf("EmptyListType"), KIND_EMPTY_LIST},
                    {TStringBuf("EmptyDictType"), KIND_EMPTY_DICT},
            };

            if (!node->IsAtom()) {
                return KIND_NONE;
            }

            auto it = map.find(node->GetContent());
            if (it != map.end()) {
                return it->second;
            }

            return KIND_ID;
        }

        bool isQuotedNode(const NYql::TAstNode *node) {
            if (!node->IsList() || !node->IsListOfSize(2)) {
                return false;
            }

            auto child = node->GetChild(0);
            return child->IsAtom() && child->GetContent() == "quote";
        }

        const NYql::TAstNode *unquoteNode(const NYql::TAstNode *node) {
            if (isQuotedNode(node)) {
                return node->GetChild(1);
            }

            return node;
        }

        TResult<TDataType> makeIDType(const NYql::TAstNode *node) {
            auto unquoted = unquoteNode(node);
            if (!unquoted->IsAtom()) {
                return TError{} << "ID node must be a atom, but got: " << unquoted->GetType();
            }

            return TDataType{
                    .Kind = EKind::KIND_ID,
                    .Value = TString(unquoted->GetContent()),
            };
        }

        TResult<TDataType> makeKeyValueType(const NYql::TAstNode *node) {
            auto unquoted = unquoteNode(node);
            if (!unquoted->IsListOfSize(2)) {
                return TError{} << "KeyValue node must have 2 children, but got: " << unquoted->GetChildrenCount();
            }

            auto keyNode = unquoteNode(unquoted->GetChild(0));
            if (!keyNode->IsAtom()) {
                return TError{} << "key must be a atom, but got: " << unquoted->GetType();
            }

            auto valueNode = ParseDataType(unquoted->GetChild(1));
            if (const auto err = GetError(valueNode)) {
                return TError{} << "unable to parse value for key " << keyNode->GetContent() << ": " << err;
            }

            return TDataType{
                    .Kind = EKind::KIND_KEY_VALUE,
                    .Value = TString(keyNode->GetContent()),
                    .Childs = {
                            std::get<0>(valueNode),
                    }
            };
        }

        TResult<TDataType> makePrimitiveType(const NYql::TAstNode *node) {
            if (node->GetChildrenCount() < 2) {
                return TError{} << "Primitive type must have more than one value";
            }

            auto value = unquoteNode(node->GetChild(1));
            if (!value->IsAtom()) {
                return TError{} << "Primitive type value must be a atom";
            }

            TVector<TDataType> childs;
            for (ui32 i = 2; i < node->GetChildrenCount(); ++i) {
                auto child = makeIDType(node->GetChild(i));
                if (IsError(child)) {
                    return child;
                }

                childs.push_back(std::get<0>(child));
            }

            return TDataType{
                    .Kind = EKind::KIND_PRIMITIVE,
                    .Value = TString(value->GetContent()),
                    .Childs = childs,
            };
        }

        TResult<TDataType> makeOptionalType(const NYql::TAstNode *node) {
            if (node->GetChildrenCount() != 2) {
                return TError{} << "Optional type must have exactly one child, but have: " << node->GetChildrenCount();
            }

            auto nested = ParseDataType(node->GetChild(1));
            if (IsError(nested)) {
                return nested;
            }

            const auto child = std::get<0>(nested);
            return TDataType{
                    .Kind = EKind::KIND_OPTIONAL,
                    .Childs = {
                            child,
                    }
            };
        }

        TResult<TDataType> makeListType(const NYql::TAstNode *node) {
            TVector<TDataType> childs;
            for (ui32 i = 1; i < node->GetChildrenCount(); ++i) {
                auto nested = ParseDataType(node->GetChild(i));
                if (const auto err = GetError(nested)) {
                    return TError{} << "unable to parse " << i << " type of List: " << err;
                }
                childs.push_back(std::get<0>(nested));
            }

            return TDataType{
                    .Kind = EKind::KIND_LIST,
                    .Childs = childs,
            };
        }

        TResult<TDataType> makeDictType(const NYql::TAstNode *node) {
            if (node->GetChildrenCount() != 3) {
                return TError{} << "Dict type must have exactly 3 childs, but have: " << node->GetChildrenCount();
            }

            TVector<TDataType> childs;
            for (ui32 i = 1; i < node->GetChildrenCount(); ++i) {
                auto nested = ParseDataType(node->GetChild(i));
                if (const auto err = GetError(nested)) {
                    return TError{} << "unable to parse " << i << " type of List: " << err;
                }
                childs.push_back(std::get<0>(nested));
            }

            return TDataType{
                    .Kind = EKind::KIND_DICT,
                    .Childs = childs,
            };
        }

        TResult<TDataType> makeTupleType(const NYql::TAstNode *node) {
            TVector<TDataType> childs;
            for (ui32 i = 1; i < node->GetChildrenCount(); ++i) {
                auto nested = ParseDataType(node->GetChild(i));
                if (const auto err = GetError(nested)) {
                    return TError{} << "unable to parse " << i << " type of Tuple: " << err;
                }
                childs.push_back(std::get<0>(nested));
            }

            return TDataType{
                    .Kind = EKind::KIND_TUPLE,
                    .Childs = childs,
            };
        }

        TResult<TDataType> makeStructType(const NYql::TAstNode *node) {
            if (node->GetChildrenCount() < 2) {
                return TError{} << "invalid struct args count: " << node->GetChildrenCount();
            }

            TVector<TDataType> childs;
            for (ui32 i = 1; i < node->GetChildrenCount(); ++i) {
                auto child = makeKeyValueType(node->GetChild(i));
                if (const auto err = GetError(child)) {
                    return TError{} << "unable to parse struct key " << i << ": " << err;
                }

                childs.push_back(std::get<0>(child));
            }

            return TDataType{
                    .Kind = EKind::KIND_STRUCT,
                    .Childs = childs,
            };
        }

        TResult<TDataType> makeVariantType(const NYql::TAstNode *node) {
            if (node->GetChildrenCount() != 2) {
                return TError{} << "Variant type must have exactly one child, but have: " << node->GetChildrenCount();
            }

            auto nested = ParseDataType(node->GetChild(1));
            if (IsError(nested)) {
                return nested;
            }

            const auto child = std::get<0>(nested);
            return TDataType{
                    .Kind = EKind::KIND_VARIANT,
                    .Childs = {
                            child,
                    }
            };
        }

        TResult<TDataType> makeVoidType(const NYql::TAstNode *node) {
            if (node->GetChildrenCount() != 1) {
                return TError{} << "unexpected Void childs count: " << node->GetChildrenCount();
            }

            return TDataType{
                    .Kind = EKind::KIND_VOID,
            };
        }

        TResult<TDataType> makeNullType(const NYql::TAstNode *node) {
            if (node->GetChildrenCount() != 1) {
                return TError{} << "unexpected Null childs count: " << node->GetChildrenCount();
            }

            return TDataType{
                    .Kind = EKind::KIND_NULL,
            };
        }

        TResult<TDataType> makeEmptyListType(const NYql::TAstNode *node) {
            if (node->GetChildrenCount() != 1) {
                return TError{} << "unexpected EmptyList childs count: " << node->GetChildrenCount();
            }

            return TDataType{
                    .Kind = EKind::KIND_EMPTY_LIST,
            };
        }

        TResult<TDataType> makeEmptyDictType(const NYql::TAstNode *node) {
            if (node->GetChildrenCount() != 1) {
                return TError{} << "unexpected EmptyDict childs count: " << node->GetChildrenCount();
            }

            return TDataType{
                    .Kind = EKind::KIND_EMPTY_DICT,
            };
        }

        TResult<TDataType> makeType(EKind kind, const NYql::TAstNode *node) {
            static const THashMap<EKind, typeMaker> map = {
                    {KIND_PRIMITIVE,  makePrimitiveType},
                    {KIND_OPTIONAL,   makeOptionalType},
                    {KIND_LIST,       makeListType},
                    {KIND_DICT,       makeDictType},
                    {KIND_TUPLE,      makeTupleType},
                    {KIND_STRUCT,     makeStructType},
                    {KIND_VARIANT,    makeVariantType},
                    {KIND_VOID,       makeVoidType},
                    {KIND_NULL,       makeNullType},
                    {KIND_EMPTY_LIST, makeEmptyListType},
                    {KIND_EMPTY_DICT, makeEmptyDictType},
            };

            auto it = map.find(kind);
            if (it != map.end()) {
                return it->second(unquoteNode(node));
            }

            return TError{} << "unexpected type kind: " << kind;
        }

        void debugPrintNode(const NYql::TAstNode *node, const TString &lvl) {
            Cout << lvl << ": " << node->GetType();
            if (node->IsAtom()) {
                Cout << "(" << node->GetContent() << ")";
            }
            Cout << Endl;

            if (isQuotedNode(node)) {
                DebugPrintNode(node, lvl);
            } else if (node->IsList()) {
                DebugPrintNode(node, lvl);
            }
        }
    }

    TResult<TDataType> ParseDataType(const NYql::TAstNode *node) {
        if (!node->IsList()) {
            return TError{} << "invalid node type " << node->GetType() << ": List expected";
        }

        if (node->GetChildrenCount() < 1) {
            return TError{} << "empty node is unexpected";
        }

        auto typ = parseTypeKind(node->GetChild(0));
        return makeType(typ, node);
    }

    void DebugPrintNode(const NYql::TAstNode *node, const TString &lvl) {
        if (node->IsAtom()) {
            debugPrintNode(node, "");
            return;
        }

        for (ui32 k = 0; k < node->GetChildrenCount(); ++k) {
            auto subChild = node->GetChild(k);
            debugPrintNode(subChild, lvl + ":" + ToString(k));
        }
    }
}