import json
import sys
import uuid
import decimal
import logging
import datetime

from six import iteritems
from dateutil import parser
from security.yql_parser import Parser, EKind
from yql.client.parameter_value_builder import \
    YqlParameterValueBuilder as ValueBuilder

logger = logging.getLogger(__name__)


class CustomExp(Exception):
    pass


def convert_primitive(query_key, parsed_ast, current_ekind, data_from_json,
                      optional_skip=False):
    """
    Function for converting primitive types.

    Args:
        query_key (str): Key from serialized json.
        parsed_ast (dict): AST like structure from parser with Ekind and type.
        current_ekind (str): Current EKind from YQL lexer.
        data_from_json (object): Data from serialized json by key.
        optional_skip (bool): Optional parameter for skipping logic.

    Returns:
        Dict: Prepared dictionary for the final YQL parameters form.
    """

    skip_to_json = False
    declare_type = parsed_ast.get('value')
    extra_type_function = extra_types_serialize.get(declare_type)

    logger.debug("Key - {}".format(query_key))
    logger.debug("Declare type - {}".format(declare_type))
    logger.debug("EKind - {}".format(kind_text.get(current_ekind)))
    logger.debug(
        "Data from json - {}".format(data_from_json))  # May leak private data

    if extra_type_function and declare_type == 'Interval':
        first_date = extra_type_function(data_from_json[0])
        second_date = extra_type_function(data_from_json[1])
        data_from_json = first_date - second_date
    elif extra_type_function:
        data_from_json = extra_type_function(data_from_json)

    if declare_type in ['TzDate', 'TzDatetime', 'TzTimestamp']:
        converted_value = data_from_json
        skip_to_json = True
    else:
        converted_value = convert_function[declare_type](data_from_json)

    if not optional_skip:
        json_ready_for_yql = {query_key: converted_value}
        result = build_json_map_with_skipper(json_ready_for_yql, skip_to_json)
        logger.debug("---------------------------------------")
    else:
        result = converted_value

    return result


def convert_optional(query_key, parsed_ast, current_ekind, data_from_json,
                     optional_skip=False):
    """
    Function for preparing optional types and calling next converting 
    functions.

    Args:
        query_key (str): Key from serialized json.
        parsed_ast (dict): AST like structure from parser with Ekind and type.
        current_ekind (str): Current EKind from YQL lexer.
        data_from_json (object): Data from serialized json by key.
        optional_skip (bool): Optional parameter for skipping logic.

    Returns:
        Dict: Prepared dictionary for the final YQL parameters form.
    """

    declare_type = parsed_ast.get('value')
    next_kind_text = kind_text.get(parsed_ast.get('childs')[0].get('kind'))

    logger.debug(
        "EKind - {} in Query Key - {}".format(kind_text.get(current_ekind),
                                              query_key))
    logger.debug("Next EKind - {}".format(next_kind_text))

    if isinstance(data_from_json, str) and len(data_from_json) == 0 \
            or not data_from_json and not isinstance(data_from_json, bool) \
            or next_kind_text == "VOID":

        logger.debug("Data from json - {}".format(
            data_from_json))  # May leak private data
        logger.debug("Declare type - {}".format(declare_type))
        logger.debug("EKind - None")

        converted_value = convert_function['Null']()
        json_ready_for_yql = {query_key: converted_value}
        result = build_json_map_with_skipper(json_ready_for_yql)

    else:
        next_kind = parsed_ast.get('childs')[0].get('kind')
        next_parsed_ast = parsed_ast.get('childs')[0]
        next_parsed_declare_type = next_parsed_ast.get('value')
        optional_value = kind_parse.get(next_kind)(query_key, next_parsed_ast,
                                                   next_kind, data_from_json,
                                                   True)

        if next_parsed_declare_type in ['TzDate', 'TzDatetime', 'TzTimestamp']:
            result = skip_convert(query_key, next_parsed_ast, next_kind,
                                  data_from_json, optional_skip=True)
        elif kind_text.get(next_kind) in [kind_text.get(key) for (key, value)
                                          in kind_parse.items() if
                                          value == skip_convert]:
            result = optional_value
        else:
            converted_value = convert_function['Optional'](optional_value)
            json_ready_for_yql = {query_key: converted_value}
            result = build_json_map_with_skipper(json_ready_for_yql)

    logger.debug("---------------------------------------")
    return result


def convert_void(query_key, parsed_ast, current_ekind, data_from_json,
                 optional_skip=False):
    """
    Function for converting empty types.

    Args:
        query_key (str): Key from serialized json.
        parsed_ast (dict): AST like structure from parser with Ekind and type.
        current_ekind (str): Current EKind from YQL lexer.
        data_from_json (object): Data from serialized json by key.
        optional_skip (bool): Optional parameter for skipping logic.

    Returns:
        Dict: Prepared dictionary for the final YQL parameters form.
    """

    logger.debug("Key - {}".format(query_key))
    logger.debug("Declare type - {}".format(parsed_ast.get('value')))
    logger.debug("EKind - {}".format(kind_text.get(current_ekind)))
    logger.debug(
        "Data from json - {}".format(data_from_json))  # May leak private data

    converted_value = convert_function['Void']()
    json_ready_for_yql = {query_key: converted_value}
    result = build_json_map_with_skipper(json_ready_for_yql)

    logger.debug("Converted value - {}".format(
        converted_value.to_json()))  # May leak private data
    logger.debug("---------------------------------------")
    return result


def skip_convert(query_key, parsed_ast, current_ekind, data_from_json,
                 optional_skip=False):
    """
    Function for unsuported types.

    Args:
        query_key (str): Key from serialized json.
        parsed_ast (dict): AST like structure from parser with Ekind and type.
        current_ekind (str): Current EKind from YQL lexer.
        data_from_json (object): Data from serialized json by key.
        optional_skip (bool): Optional parameter for skipping logic.

    Returns:
        Dict: Prepared dictionary for the final YQL parameters form.
    """

    logger.debug("Key - {}".format(query_key))
    logger.debug("Declare type - {}".format(parsed_ast))
    logger.debug("EKind - {}".format(kind_text.get(current_ekind)))

    if optional_skip:
        converted_value = [data_from_json]
        json_ready_for_yql = {query_key: converted_value}
        result = build_json_map_with_skipper(json_ready_for_yql,
                                             skip_to_json=True)
    else:
        converted_value = data_from_json
        json_ready_for_yql = {query_key: converted_value}
        result = build_json_map_with_skipper(json_ready_for_yql,
                                             skip_to_json=True)

    logger.debug("Converted value: {}\n".format(converted_value))  # Delete
    logger.debug("---------------------------------------")
    return result


def build_json_map_with_skipper(valuesDict, skip_to_json=False):
    """
    Returns converted to final yql format dict.

    Args:
        valuesDict (dict): Dict with values.
        skip_to_json (bool): optional value for skip logic.

    Returns:
        Dict: Converted.
    """

    if skip_to_json:
        result = {k: json.dumps({'Data': v}) for k, v in iteritems(valuesDict)}
    else:
        result = {k: json.dumps({'Data': v.to_json()}) for k, v in
                  iteritems(valuesDict)}

    return result


# supporting type converting
kind_parse = {
    EKind.KIND_PRIMITIVE: convert_primitive,
    EKind.KIND_OPTIONAL: convert_optional,
    EKind.KIND_VOID: convert_void,
    EKind.KIND_NULL: convert_void,
    EKind.KIND_NONE: skip_convert,
    EKind.KIND_LIST: skip_convert,
    EKind.KIND_DICT: skip_convert,
    EKind.KIND_TUPLE: skip_convert,
    EKind.KIND_STRUCT: skip_convert,
    EKind.KIND_VARIANT: skip_convert,
    EKind.KIND_EMPTY_LIST: skip_convert,
    EKind.KIND_ID: skip_convert,
    EKind.KIND_EMPTY_DICT: skip_convert,
    EKind.KIND_KEY_VALUE: skip_convert
}

kind_text = {
    0: "NONE",
    1: "PRIMITIVE",
    2: "OPTIONAL",
    100: "LIST",
    101: "DICT",
    102: "TUPLE",
    103: "STRUCT",
    104: "VARIANT",
    200: "VOID",
    201: "NULL",
    202: "EMPTY_LIST",
    203: "EMPTY_DICT",
    300: "ID",
    301: "KEY_VALUE"
}

# supporting extra type converting
extra_types_serialize = {
    'Decimal': decimal.Decimal,
    'Uuid': uuid.UUID,
    'Date': parser.parse,
    'Datetime': parser.parse,
    'Timestamp': parser.parse,
    'Interval': parser.parse
}

# supporting yql library converting
convert_function = {
    'Int8': ValueBuilder.make_int8,
    'Int16': ValueBuilder.make_int16,
    'Int32': ValueBuilder.make_int32,
    'Int64': ValueBuilder.make_int64,
    'Uint8': ValueBuilder.make_uint8,
    'Uint16': ValueBuilder.make_uint16,
    'Uint32': ValueBuilder.make_uint32,
    'Uint64': ValueBuilder.make_uint64,
    'Bool': ValueBuilder.make_bool,
    'Float': ValueBuilder.make_float,
    'Double': ValueBuilder.make_double,
    'Decimal': ValueBuilder.make_decimal,
    'String': ValueBuilder.make_string,
    'Utf8': ValueBuilder.make_utf8,
    'Uuid': ValueBuilder.make_uuid,
    'Yson': ValueBuilder.make_yson_from_str,
    'Json': ValueBuilder.make_json,
    'Date': ValueBuilder.make_date,
    'Datetime': ValueBuilder.make_datetime,
    'Timestamp': ValueBuilder.make_timestamp,
    'Interval': ValueBuilder.make_interval,
    'Void': ValueBuilder.make_void,
    'Null': ValueBuilder.make_null,
    'Optional': ValueBuilder.make_optional,
}


def main(query, data_from_json):
    """
    Function for parsing query and converting data from json to yql format.

    Args:
        query (str): Query with prepared statement - Declare.
        data_from_json (object): Serialized json with data for query.

    Returns:
        Dict: Prepared dictionary for the final YQL parameters form.
    """

    logger.info("Starting parsing and converting...")

    # Parsing YQL query with python binding
    result = {}
    parser = Parser()
    parsed_query = parser.parse(query)

    logger.debug("Parsed query: \n {}".format(parsed_query))
    logger.debug("Serialized json: \n {}".format(
        data_from_json))  # May leak private data
    logger.debug("---------------------------------------")

    # conversion process
    for key, value in parsed_query.items():
        convert_function = kind_parse.get(value.get('kind'))
        single_data = data_from_json.get(key)

        if not convert_function:
            logger.info("Find unknown EKind")

        if key not in data_from_json:
            raise CustomExp(
                "Declared parameter not found in Json. Key - {}".format(key))

        result.update(
            convert_function(key, value, value.get('kind'), single_data))

    logger.debug(
        "Converted result: \n {}".format(result))  # May leak private data
    logger.info("Parsing and converting ended.")
    return result
