import sys

try:
    __import__('pkg_resources').require('requests')
except Exception:
    if getattr(sys, 'is_standalone_binary', False):
        raise

import requests


class ConductorGroupNotFoundError(Exception):
    def __init__(self, group_name):
        super(ConductorGroupNotFoundError, self).__init__("conductor group '{}' not found".format(group_name))


class ConductorTagNotFoundError(Exception):
    def __init__(self, tag_name):
        super(ConductorTagNotFoundError, self).__init__("conductor tag '{}' not found".format(tag_name))


class ConductorProjectNotFoundError(Exception):
    def __init__(self, tag_name):
        super(ConductorProjectNotFoundError, self).__init__("conductor project '{}' not found".format(tag_name))


class ConductorDCNotFoundError(Exception):
    def __init__(self, tag_name):
        super(ConductorDCNotFoundError, self).__init__("conductor data center '{}' not found".format(tag_name))


class ConductorHttpError(Exception):
    def __init__(self, code, reason):
        self.code = code
        self.reason = reason
        if code:
            super(ConductorHttpError, self).__init__('conductor api http error: {0} {1}'.format(code, reason))
        else:
            super(ConductorHttpError, self).__init__('conductor api http error: {0}'.format(reason))


class Conductor(object):
    API_URL = 'http://c.yandex-team.ru/api/{0}/{1}'
    DEFAULT_TIMEOUTS = [1, 4, 8, 16]

    def __init__(self, api_url=None, timeouts=None):
        self.timeouts = self.DEFAULT_TIMEOUTS
        if timeouts is not None:
            self.timeouts = timeouts
        self.api_url = self.API_URL
        if api_url is not None:
            self.api_url = api_url

    def hosts(self, group_name, method='groups2hosts', exception=ConductorGroupNotFoundError):
        exc = None
        for timeout in self.timeouts:
            try:
                r = requests.get(self.api_url.format(method, group_name), timeout=timeout)
                exc = None
                break
            except requests.exceptions.Timeout:
                exc = 'timeout'
            except requests.exceptions.ConnectionError as e:
                exc = str(e)

        if exc is not None:
            raise ConductorHttpError(0, exc)

        if r.status_code != requests.codes.ok:
            if r.status_code == requests.codes.not_found:
                raise exception(group_name)
            # r.reason was not available in this version of requests, that's why
            raise ConductorHttpError(r.status_code, "")

        hosts = r.text.splitlines()
        return hosts

    def hosts_by_group(self, group_name):
        return self.hosts(group_name, 'groups2hosts', ConductorGroupNotFoundError)

    def hosts_by_tag(self, tag_name):
        return self.hosts(tag_name, 'tag2hosts', ConductorTagNotFoundError)

    def hosts_by_project(self, project_name):
        return self.hosts(project_name, 'projects2hosts', ConductorProjectNotFoundError)

    def hosts_by_dc(self, dc_name):
        return self.hosts(dc_name, 'dc2hosts', ConductorDCNotFoundError)
