"""
Config
~~~~~~

Global skynet config.

Config variables
================

portoffset
----------

The base port offset all skynet services is bind to.
Default: 10000. Example:

.. code-block:: ini

   portoffset = 12345

copier_transport
----------------

Copier transport to use.
Default: ``'torrent'``

This will be used primarily for sky download/upload/share commands, which
call api.copier.Copier().create() automatically.

If you call Copier().create() by yourself and dont specify
transport= as arg, default transport will be used and this setting
does not affect it at all. The same if you force transport then creating
resource via transport= -- your enforcement value will be used, does not
matter what is set in ~/.config.

Example:

.. code-block:: ini

   copier_transport = rbtorrent
"""

import os


class Config_(object):
    def __init__(self):
        self.portOffset = int(os.environ.get('SKYNET_PORTOFFSET', 0))
        self.copierTransport = 'rbtorrent'
        self.useGevent = False

        try:
            configDir = os.path.expanduser('~/.skynet')

            if os.path.isfile(configDir):
                import warnings
                warnings.warn('Using ~/.skynet as file is deprecated, please rename it to ~/.skynet/user.conf!')
                confFile = configDir
            else:
                confFile = os.path.join(configDir, 'user.conf')

            if os.path.exists(confFile):
                for line in open(confFile, 'r'):
                    (key, value) = [x.strip() for x in line.split('=')]

                    if key == 'portoffset' and not os.environ.get('SKYNET_FORCE_PORTOFFSET', False):
                        self.portOffset = int(value)
                    elif key == 'copier_transport':
                        self.copierTransport = value
                    elif key == 'use_gevent':
                        self.useGevent = True if value in ('1', 'yes', 'True', 'true') else False
        except BaseException:
            pass


Config = Config_()


def basePortOffset():
    """
    Returns define base port offset.

    :rtype: int
    """
    return Config.portOffset


def copierTransport():
    """
    Returns defined copier transport.

    :rtype: string
    """
    return Config.copierTransport


def useGevent(flag=None):
    """
    Tell all libraries to use gevent if they able to do that.
    """
    if flag is not None:
        try:
            return Config.useGevent
        finally:
            Config.useGevent = flag
    return Config.useGevent
