'''
Description: Public API for Skynet.Copier service.
'''

import os
import warnings
import types

from api.copier.errors import (  # noqa
    CopierError,
    ApiError, ApiConnectionError,
    FilesystemError, Timeout,
    UnshareableResource,
    ResourceError, ResourceNotAvailable, ResourceNotAllowedByNetwork, ResourceDownloadError,
    NotRegisteredJob
)
from api.copier.utils import ensureDest, sanitizeCWD, convertFileList

from kernel.util.functional import memoized, threadsafe


class Enumerator(object):
    @classmethod
    def toString(cls, val):
        try:
            return dict(
                (klass, klassName)
                for klassName, klass in (
                    (name, getattr(cls, name))
                    for name in dir(cls)
                )
                if isinstance(klass, types.ClassType)
            )[getattr(cls, val, None) if isinstance(val, basestring) else val]
        except KeyError:
            raise ApiError('Invalid value specified: %r' % (val, ))


class Priority(Enumerator):
    class Idle:
        pass

    class Low:
        pass

    class BelowNormal:
        pass

    class Normal:
        pass

    class AboveNormal:
        pass

    class High:
        pass

    class RealTime:
        pass


class Network(Enumerator):
    class Auto:
        pass

    class Backbone:
        pass

    class Fastbone:
        pass


class Deduplicate(Enumerator):
    class No:
        pass

    class Symlink:
        pass

    class Hardlink:
        pass


def _get_default_network():
    from library.config import query
    return getattr(Network, query('skynet.services.copier').network.mode)


class IGetResult(object):
    def __init__(self, files):
        self.__files = files

    def files(self):
        """
        Get list of files

        :rtype: list
        """
        return self.__files


class IGet(object):
    def __init__(self, resid, slave):
        self.__resid = resid
        self.__slave = slave

    def wait(self, timeout=None):
        """
        Wait until download completes.

        :arg timeout: optional timeout
        :type timeout: float (seconds) or None

        :rtype: :py:class:`.IGetResult`
        """
        return IGetResult(self.__slave.wait(timeout=timeout))

    def iter(self, timeout=None, state_version=1):
        for res in self.__slave.iter(timeout=timeout, state_version=state_version):
            yield res

    def stop(self):
        return self.__slave.stop()

    def resid(self):
        """
        Get :term:`resource id` associated with handle.
        """
        return self.__resid


class IListResult(IGetResult):
    """
    Proxy for IGetResult then .list() is invoked. Backward-compatible transition to return IGetResult instead
    of simple list with dictionaries.
    """

    def _warn(self):
        warnings.warn('You must not use result of list() as list, use it as IGetResult (e.g. via .files())')

    def __getitem__(self, idx):
        self._warn()
        return self.files()[idx]

    def __setitem__(self, idx, value):
        self._warn()
        self.files()[idx] = value

    def __delitem__(self, idx):
        self._warn()
        del self.files()[idx]

    def __contains__(self, value):
        self._warn()
        return value in self.files()

    def __getattr__(self, attr):
        self._warn()
        return getattr(self.files(), attr)

    def __iter__(self):
        self._warn()
        return iter(self.files())

    def __len__(self):
        self._warn()
        return len(self.files())


class IList(object):
    def __init__(self, resid, slave):
        self.__resid = resid
        self.__slave = slave

    def iter(self, timeout=None, state_version=1):
        for res in self.__slave.iter(timeout=timeout, state_version=state_version):
            yield res

    def wait(self, timeout=None):
        """
        Wait until operation will complete.

        :rtype: :py:class:`.IListResult`
        """
        return IListResult(self.__slave.wait(timeout=timeout))

    def resid(self):
        """
        Get associated :term:`resource id`.

        :rtype: :term:`resource id` (string)
        """
        return self.__resid


class IHandle(object):
    def __init__(self, resid, slave):
        self.__slave = slave
        self.__resid = resid

    def resid(self):
        """
        Get :term:`resource id` associated with handle.

        :rtype: :term:`resource id` (string)
        """
        return self.__resid

    def get(
        self, dest=None, user=False,
        priority=Priority.Normal, network=None, deduplicate=Deduplicate.No,
        max_dl_speed=None, max_ul_speed=None,
        **extra
    ):
        """
        Download :term:`resource` to dest directory.

        :arg dest: Destination directory
        :type dest: string or None
        :arg bool user: Flags either run the transport in user mode or not.
                        Note that `rbtorrent` transport needs a list of resource files to pre-create
                        them __before__ the download will start, so the client will try to fetch
                        files list first. To correctly timeout that operation you should call
                        `handle.list().wait(timeout)` prior to `handle.get()` method call.
        :rtype: :py:class:`.IGet`
        """

        if network is None:
            network = _get_default_network()

        limits = [max_dl_speed, max_ul_speed]
        for idx, value in enumerate(limits):
            if not value:
                continue
            value = value.lower()
            try:
                value = int(value)
            except:
                if value.endswith('mbps'):
                    value = int(int(value[:-4]) * 10**6 // 8)
                elif value.endswith('m'):
                    value = int(int(value[:-1]) * 10**6)
                elif value.endswith('mb'):
                    value = int(int(value[:-2]) * 10**6)
                else:
                    raise

            limits[idx] = value

        return IGet(
            self.resid(),
            self.__slave.get(
                dest=ensureDest(dest, perms=0o755 if user else 0o777),
                user=user,
                priority=priority,
                network=network,
                deduplicate=deduplicate,
                max_dl_speed=limits[0],
                max_ul_speed=limits[1],
                **extra
            )
        )

    def list(
        self, priority=Priority.Normal, network=None, **extra
    ):
        """
        List files in associated :term:`resource`.

        :rtype: :py:class:`.IList`
        """
        if network is None:
            network = _get_default_network()

        return IList(
            self.resid(),
            self.__slave.list(
                priority=priority,
                network=network,
                **extra
            )
        )


class ICopier(object):
    Priority = Priority

    def __init__(self, slave):
        self.__slave = slave

    def create(self, files, cwd=None, transport=None, **extra):
        """
        Create new :term:`resource` from list of files with possible CWD.

        :arg files: Source paths for new :term:`resource`
        :type files: iterable

        :arg cwd: Change workdir before searching files
        :type cwd: string or None

        :arg unknown transport: **DO NOT USE**, internal only, will be removed/changed soon

        :rtype: :py:class:`ya.skynet.services.copier.transports.rbtorrent.ResourceHandler`

        .. warning::
            do not use ``transport`` argument -- it is for internal use only.

        .. rubric:: Argument: ``files``

        Should be a list of:

          a) file names (e.g. ``/etc/hosts``)
          b) tuples of file path and wanted path in :term:`resource`. E.g. ``('/etc/hosts', 'etc/hosts_file')``.
             Note that in this case file path **must be** absolute, relative will not work. Also cwd is ignored
             for these paths.

        You can mix types ``a`` and ``b`` during creation of one :term:`resource`.

        .. rubric:: Argument: ``cwd``

        If you specify ``cwd``, copier will ``cd`` to that directory before searching
        files. ``None`` value is identical to ``'.'`` (dot).

        .. rubric:: Example

        >>> from api.copier import Copier
        >>> copier = Copier()
        >>> handle = copier.create([
        ...     '/etc/hosts',
        ...     ('/etc/resolv.conf', 'etc/resolv_conf_file')
        ... ])
        >>> print(handle.resid())  # this will print generated resource id

        In the example above we :term:`resource` will contain two files:
         - ``etc/hosts`` (from ``/etc/hosts``)
         - ``etc/resolv_conf_file`` (from ``/etc/resolv.conf``)
        """
        if cwd:
            cwd = sanitizeCWD(cwd)
        return self.createEx(files, cwd, transport, **extra).wait()

    def createEx(self, files, cwd=None, transport=None, **extra):
        """
        Async version of create(). It returns handle which has .wait() method which
        blocks until :term:`resource` will be created.

        :arg files: Source paths for new :term:`resource`
        :type files: iterable

        :arg cwd: Change workdir before searching files
        :type cwd: string or None

        :arg unknown transport: **DO NOT USE**, internal only, will be removed/changed soon

        :rtype: :py:class:`ya.skynet.services.copier.transports.rbtorrent.ResourceHandler`
        """

        if cwd:
            cwd = sanitizeCWD(cwd)

        realFiles = []

        # Read all symlinks, since createEx must do that by default.
        for (realPath, virtualPath) in convertFileList(files, cwd, traverse_symlinks=True):
            # We need to check here for file/folder existance, because after this
            # step original symlink path will be unavailable.
            # This requires calling user be able to list contents in target dirs, which
            # is not 100% requirement, since they *must* be readable only by skynet
            # user.
            # But, since this is anyway nice security hole... (share file which is not readable
            # by you, but readable by skynet, and then, ask sky to get it somewhere you will
            # be able to read... thats it!) -- we dont bother about this.
            # createExEx() will not check files here.
            if os.path.islink(realPath):
                realRealPath = os.path.realpath(realPath)
                if not os.path.exists(realRealPath):
                    raise FilesystemError('Symbolic link %r points to non-existent file or folder %r' % (
                        realPath, realRealPath
                    ))
            else:
                if not os.path.exists(realPath):
                    raise FilesystemError('File or folder %r does not exist.' % realPath)
                realRealPath = realPath

            realFiles.append((realRealPath, virtualPath))

        return self.__slave.create(realFiles, transport=transport, **extra)

    def createExEx(self, files, cwd=None, transport=None, **extra):
        """
        createEx(), but saves symlinks as-is (they will be stored as symlinks
        in :term:`resource` and will be recreated on downloading machines.

        :TODO: return ICreate interface object

        :arg files: Source paths for :term:`resource`
        :type files: iterable

        :arg cwd: Change workdir before searching files
        :type cwd: string or None

        :arg unknown transport: **DO NOT USE**, internal only, will be removed/changed soon

        :rtype: :py:class:`ya.skynet.services.copier.transports.rbtorrent.ResourceHandler`
        """
        if cwd:
            cwd = sanitizeCWD(cwd)
        return self.__slave.create(convertFileList(files, cwd), transport=transport, **extra)

    def handle(self, resid):
        """
        Get handle for :term:`resource`

        :arg string resid: :term:`resource id`
        :rtype: :py:class:`.IHandle`
        """
        return IHandle(resid, self.__slave.handle(resid))

    def get(
        self, resid, dest=None, user=True,
        priority=Priority.Normal, network=None, deduplicate=Deduplicate.No,
        max_dl_speed=None, max_ul_speed=None,
        **extra
    ):
        """
        Shortcut to ``self.handle(resid).get(dest)``.

        :arg string resid: :term:`resource id`

        :arg dest: destination path on this machine
        :arg bool user: flags either run the transport in user mode or not
        :type dest: string or None

        :rtype: :py:class:`.IGet`
        """
        if network is None:
            network = _get_default_network()
        return self.handle(resid).get(
            dest, user, priority, network, deduplicate,
            max_dl_speed, max_ul_speed, **extra
        )

    def list(
        self, resid, timeout=None,
        priority=Priority.Normal, network=None,
        **extra
    ):
        """
        Shortcut to ``self.handle(resid).list().wait()``.

        :arg string resid: :term:`resource id`
        :rtype: list of dicts with keys

            :name: file path in resource (e.g. "my file" or "path/to/my file")
            :size: size in bytes
            :executable: is file executable or not (True/False)
            :type: file, dir or symlink
            :md5sum: md5 checksum of whole file

        """
        if network is None:
            network = _get_default_network()
        return self.handle(resid).list(priority, network, **extra).wait(timeout=timeout)

    def copy(
        self,
        srcHost, srcPath, tgtHost, tgtPath,
        priority=Priority.Normal, network=Network.Backbone,
        srcUser=None, tgtUser=None
    ):
        """
        Copy files directly using KQueue (fast)

        :arg string srcHost: host to copy from

        :arg string srcPath: full file path on the source host

        :arg string tgtHost: host to copy to

        :arg string tgtPath: full path to file or folder on the target host

        :arg priority: ignored

        :arg string srcUser: optional username to authenticate on the source machine as.
             By default current user is used
        :arg string tgtUser: optional username to authenticate on the target machine as.
             By default current user is used
        """
        network_text = Network.toString(network)
        if network_text != 'Backbone':
            raise ApiError('copy() supports only Backbone network')
        if not srcUser and not tgtUser:
            raise ApiError('Either tgtUser or srcUser are required to be specified for scp transport')
        return self.__slave.copy(srcHost, srcPath, tgtHost, tgtPath, priority, srcUser, tgtUser)


@threadsafe
@memoized
def copierClass():
    from api.skycore import ServiceManager
    return ServiceManager().get_service_python_api('skynet', 'skybone')


def Copier():  # pylint: disable-msg=C0103
    """
    Get interface to Skynet Copier service

    :rtype: :py:class:`.ICopier`
    """
    skybone_instance = copierClass()
    if isinstance(skybone_instance, type):
        skybone_instance = skybone_instance()
    return ICopier(skybone_instance)
