import os
import py

from py.path import local as LocalPath  # pylint: disable-msg=E0611,F0401

from api.copier import errors


def convertFileList(files, cwd, traverse_symlinks=False):
    """
    This will convert simple file list [file1, file2, ...] to extended
    version using possible cwd:

    This function should *never* read symlinks (e.g. dont call os.path.realpath()
    here)!

    TODO: test this
    """

    if cwd:
        cwd = cwd.rstrip('/')

    result = []

    # Transform files to the new form
    # [(absolutePath, pathInResource), ...]
    for item in files:
        if not isinstance(item, (list, tuple)):
            # Maybe already provided in new form? :)
            # Do nothing here in this case.

            if not os.path.isabs(item):
                # file a/b, cwd /x/y --> file /x/y/a/b as a/b
                # file a/b, cwd None --> file ./a/b as realPath(.)/a/b
                if cwd:
                    absolute_path = os.path.join(cwd, item)
                else:
                    absolute_path = os.path.join(os.path.realpath('.'), item)
                path = item
            else:
                # file /a/b, cwd /x/y --> file /a/b as a/b
                # file /a/b/c, cwd /a --> file /a/b/c as b/c
                # file /a/b, cwd None --> file /a/b as a/b
                absolute_path = item
                if cwd:
                    path = absolute_path[len(os.path.commonprefix((absolute_path, cwd))):]
                else:
                    path = absolute_path

            path = path.lstrip('/')
            item = (absolute_path, path)

        real_path, target_path = item
        real_path = LocalPath(real_path)

        result.append((real_path.strpath, target_path))

        def _flt(x):
            if x.check(link=1):
                return traverse_symlinks

            return True

        symlink_inodes = {}
        checked_parts = set()

        if real_path.check(dir=1) and (traverse_symlinks or not real_path.check(link=1)):
            for real_sub_path in real_path.visit(rec=_flt):
                if traverse_symlinks:
                    # Check each symlink, maybe we would hit it twice (that will mean it is cycled)
                    for part in real_sub_path.parts():
                        if part not in checked_parts:
                            if part.check(link=1, dir=1):
                                ino = part.stat().ino
                                if ino in symlink_inodes:
                                    raise errors.FilesystemError(
                                        'Found cyclic symlink: %s' % (symlink_inodes[ino], )
                                    )
                                else:
                                    symlink_inodes[ino] = part
                            checked_parts.add(part)

                result.append((real_sub_path.strpath, os.path.join(target_path, real_sub_path.relto(real_path))))

    return result


def sanitizeCWD(cwd):
    """
    Check and sanitize provided cwd. Can raise ApiError.
    """

    cwd = cwd.rstrip('/')
    cwd = os.path.realpath(cwd)  # ensure we read symlink

    if not os.path.isabs(cwd):
        raise errors.ApiError('cwd (%r) must be absolute path' % cwd)
    if not os.path.exists(cwd):
        raise errors.ApiError('cwd (%r) does not exist' % cwd)
    if not os.path.isdir(cwd):
        raise errors.ApiError('cwd (%r) must be a directory' % cwd)
    return cwd


def ensureDest(dest, perms):
    if not dest:
        dest = os.path.realpath('.')
    else:
        # This will convert any non-absolute path to real
        # Also it will resolve any symlinks to realpath
        dest = os.path.realpath(dest)

    for part in LocalPath(dest).parts():
        if part.check(exists=1, dir=1):
            continue

        if part.check(exists=1, dir=0):
            raise errors.CopierError('Path %r is not a directory!' % part.strpath)

        # noinspection PyUnresolvedReferences
        try:
            part.ensure(dir=1)
        except py.error.Error as err:  # pylint: disable-msg=E1101
            raise errors.CopierError('Destination directory %r does not exist, failed to create: %r' % (part, err))

        # noinspection PyUnresolvedReferences
        try:
            part.chmod(perms)
        except py.error.Error as err:  # pylint: disable-msg=E1101
            raise errors.CopierError('Failed to set 0%o permissions on path %r: %s' % (perms, part, err))

    return dest
