# -*- coding: utf-8 -*-

import os
import sys
import warnings

try:
    from kernel.util.functional import memoized
except ImportError:
    def memoized(fn):
        return fn

try:
    from api.config import useGevent
except ImportError:
    def useGevent(flag=None):
        return False


from .session import ICQSession
from .sign import ICQSigner
from .primitives import ICQQueue, ICQPipe
from .exceptions import _wrapException
from .poll import ICQPoll

__all__ = [
    'ICQClient',
    'Client',
]


class ICQClient(object):
    """
    Локальный клиент, который позволяет отправлять задачи на кластер и отслеживать их выполнение.

    Клиент поддерживает Python-семантику ``with``: при выходе из ``with``-блока будет автоматически вызван метод
    :meth:`~.ICQClient.shutdown`, завершающий все запущенные сессии и останавливающий выполнение всех ещё не завершившихся задач:

    .. code-block:: py

        with client:
            session = client.run(hosts, task)
            # Do some work
        # После выхода из блока with мы можем быть уверены, что сессия session и все её задачи прекратили своё существование
    """
    def __init__(self, slave, implementation, args, kwargs):
        self.__slave = slave
        self.__implementation = implementation
        self.__args = args
        self.__kwargs = kwargs
        _check_vulnerable_keys(self.signer)

    def __reduce__(self):
        return _reconstruct_client, (self.__implementation, self.__args, self.__kwargs)

    def createQueue(self, maxsize=0):
        try:
            return ICQQueue(self.__slave.createQueue(maxsize=maxsize))
        except:
            _wrapException()
            raise

    def create_pipe(self):
        """
        Создание трубы для использования в задачах.

        .. versionchanged:: 17.0.0
            Старое название метода: createPipe

        :return: :class:`~.primitives.ICQPipe`
        """
        try:
            return ICQPipe(self.__slave.createPipe())
        except:
            _wrapException()
            raise

    createPipe = create_pipe

    def create_poll(self, iterable=None):
        """
        Создание поллера для опроса сразу нескольких
        источников на наличие данных.

        .. NOTE::

            Каждый объект при добавлении в поллер создаёт два служебных
            файловых дескриптора. Если таких объектов будет много, то
            есть риск упереться в лимит количества fd для процесса.

        .. versionchanged:: 17.0.0
            Старое название метода: createPoll

        :param iterable: если указан, то поллер сразу создаётся
                         с начальным списком объектов для опроса
        :return: :class:`~.poll.ICQPoll`
        """
        try:
            return ICQPoll(self.__slave.createPoll(iterable))
        except:
            _wrapException()
            raise

    createPoll = create_poll

    def run(self, hosts, remoteObject, params=None):
        """
        Исполняет пользовательскую функцию или объект на указанном наборе хостов.
        В качестве результата возвращает вновь созданную сессию, ответственную за
        исполнение и позволяющую получать результаты исполнения.

        Смотри пример :ref:`reference_examples_load_avg`

        :param hosts: либо отдельный хост, либо итерируемый список хостов
        :param remoteObject: :func:`callable`, который будет исполняться (объект с методом
                             :meth:`~object.__call__` или функция)
        :param params: если указан, то задаёт параметры, передаваемые удалённому
                       объекту в зависимости от хоста:
                       в ``remoteObject`` на ``hosts[i]`` будет передан параметр ``params[i]``.
                       Если ``hosts`` не является списком, то ``params`` рассматривается и
                       передаётся как один параметр.
        :return: Созданная сессия, позволяющая взаимодействовать с удалёнными узлами
                 (собирать результаты и т.п.)
        :rtype: :class:`~.session.ICQSession`
        """
        try:
            return ICQSession(self.__slave.run(hosts, remoteObject, params))
        except:
            _wrapException()
            raise

    def ping(self, hosts, checkFreeSpace=True, checkWritable=False):
        """
        Служебный вызов для проверки жизнеспособности списка хостов.
        Не требует, в отличие от обычных задач, авторизации для доступа к машине.
        Созданная сессия в качестве результата возвращает стандартные тройки значений
        (см. :meth:`.ICQSession.wait`), где результат всегда равен :data:`True` или :data:`False`
        (если проверка провалена), а в поле exception лежит исключение, если проверка
        была провалена.

        :param hosts: либо отдельный хост, либо итерируемый список хостов
        :param bool checkFreeSpace: необходимо ли проверять наличие свободного места в рабочем каталоге
                                    cqudp. Задача считается проваленной, если количество свободного места
                                    меньше 500KiB.
        :param bool checkWritable: необходимо ли проверять все файлы в рабочем каталоге cqudp. Проверяется
                                   наличие RW-доступа к каждому файлу в рабочем каталоге, проверка считается
                                   проваленной, если к хотя бы одному файлу такого доступа нет.
        :return: Созданная сессия, позволяющая взаимодействовать с удалёнными узлами
                 (собирать результаты и т.п.)
        :rtype: :class:`~.session.ICQSession`
        """
        try:
            return ICQSession(self.__slave.ping(hosts, checkFreeSpace=checkFreeSpace, checkWritable=checkWritable))
        except:
            _wrapException()
            raise

    def iter(self, hosts, remoteObject, params=None):
        """
        Исполняет задачу на выбранных хостах и возвращает результаты удалённо итерируемых генераторов.
        Поддерживает только прямые итераторы.

        :param hosts: либо отдельный хост, либо итерируемый список хостов
        :param remoteObject: генератор для итерации
        :param params: если указан, то задаёт параметры, передаваемые удалённому объекту в
                       зависимости от хоста: в ``remoteObject`` на ``hosts[i]`` будет передан
                       параметр ``params[i]``. Если ``hosts`` не является списком,
                       то ``params`` рассматривается и передаётся как один параметр.
        :return: Созданная сессия, позволяющая взаимодействовать с удалёнными узлами
                 (собирать результаты и т.п.)
        :rtype: :class:`~.session.ICQSession`
        """
        try:
            return ICQSession(self.__slave.iter(hosts, remoteObject, params))
        except:
            _wrapException()
            raise

    def iterFull(self, hosts, remoteObject):
        try:
            return ICQSession(self.__slave.iterFull(hosts, remoteObject))
        except:
            _wrapException()
            raise

    def run_in_porto(self, hosts, remoteObject, params=None):
        """
        Исполняет задачу в porto-контейнерах на выбранных хостах.
        Внутри заданного контейнера будет создан временный подконтейнер, в котором и
        будет исполнена задача.

        :param iterable hosts: список пар вида ``(hostname, container_name)``, где:

            * ``hostname``: хост, может быть строкой с необязательным указанием порта
              (``"host:port"``) или парой ``(host, port)``
            * ``container_name``: имя контейнера, в котором исполнить задачу

        :param remoteObject: :func:`callable`, который будет исполняться (объект с методом
                             :meth:`~object.__call__` или функция)
        :param params: если указан, то задаёт параметры, передаваемые удалённому
                       объекту в зависимости от хоста:
                       в ``remoteObject`` на ``hosts[i]`` будет передан параметр ``params[i]``.
                       Каждый параметр должен быть словарём с двумя опциональными ключами:

                           * optional dict ``'porto_params'``: опции для порто-контейнера. Возможные опции смотрите в `portoctl`.
                           * optional list ``'task_params'``: аргументы, с которыми запускается задача на удалённой стороне.

                       Пример::

                           params=[
                               {
                                   'porto_params': {
                                       'cpu_priority': 'rt',
                                       'memory_guarantee': 256 * 1024 * 1024,
                                   }
                               },
                               {
                                   'task_params': [1, 2, []]
                               },
                               {},
                               {
                                   'porto_params': {
                                       'cpu_priority': 'rt',
                                   },
                                   'task_params': ['param1']
                               }
                           ]
        :return: Созданная сессия, позволяющая взаимодействовать с удалёнными узлами (собирать результаты и т.п.)
        :rtype: :class:`~.session.ICQSession`
        """
        try:
            return ICQSession(self.__slave.run_in_porto(hosts, remoteObject, params=params))
        except:
            _wrapException()
            raise

    def run_in_portoshell(self, hosts, command, user=None, streaming=False, **kwargs):
        """
        Исполняет задачу в portoshell-слотах.
        Внутри контейнера слота будет создан временный подконтейнер, в котором и будет исполнена шелл-команда.

        :param iterable hosts: список кортежей вида ``(hostname, slotname)`` или
                               ``(hostname, slotname, configuration_id)``
        :param str command: шелл-команда
        :param str user: имя пользователя, под которым должна быть исполнена команда.
                         Если не указано, то в зависимости от вида изоляции контейнера
                         будет использован пользователь root или тот пользователь,
                         под которым запущен контейнер.
        :param bool streaming: использовать ли стриминговый режим с посылкой каждой
                               строки вывода одним чанком
        :return: Созданная сессия, позволяющая взаимодействовать с удалёнными узлами (собирать результаты и т.п.)
        :rtype: :class:`~.session.ICQSession`
        """
        try:
            try:
                slave = self.__slave.run_in_portoshell(hosts, command, user=user, streaming=streaming, is_iter=streaming, **kwargs)
            except TypeError:
                # backward compatibility with cqudp pre 2.1.6
                try:
                    slave = self.__slave.run_in_portoshell(hosts, command, user=user, streaming=streaming, is_iter=streaming)
                except TypeError:  # backward compatibility with cqudp pre 1.4.24
                    slave = self.__slave.run_in_portoshell(hosts, command, user=user, streaming=streaming)
            return ICQSession(slave)
        except:
            _wrapException()
            raise

    def register_safe_unpickle(self, module_name=None, attr_name=None, obj=None):
        """
        Регистрирует нестандартный тип, который серверу разрешено нам присылать.
        Требует указания либо пары ``module_name`` и ``attr_name``, либо
        параметра ``obj``.

        :param str module_name: полное квалифицированное имя модуля
        :param str attr_name: каноническое имя класса или функции внутри модуля
        :type obj: function or type
        :param obj: класс или функция, которые надо разрешить. Имя модуля и самого объекта будут извлечены автоматически

        .. seealso::

            :ref:`Возвращаемые типы данных <unsafe-types-unpickling>`
                Разрешённые по умолчанию типы
        """
        try:
            self.__slave.register_safe_unpickle(module_name=module_name, attr_name=attr_name, obj=obj)
        except:
            _wrapException()
            raise

    def shutdown(self):
        """
        Завершить работу данного клиента и всех созданных им сессий.

        :meth:`~.ICQClient.shutdown` автоматически будет вызван при выходе из блока ``with``
        или при сборе клиента сборщиком мусора.
        """
        try:
            self.__slave.shutdown()
        except:
            _wrapException()
            raise

    @property
    def running(self):
        """
        Признак того, что клиент работает. Сразу после создания этот признак выставлен в :data:`True`.
        Сбрасывается после явного вызова :meth:`~.ICQClient.shutdown`

        :rtype: bool
        """
        try:
            return self.__slave.running
        except:
            _wrapException()
            raise

    @property
    def signer(self):
        """
        Менеджер ключей, используемый данным клиентом для подписи задач

        :rtype: :class:`.ICQSigner`
        """
        try:
            return ICQSigner(self.__slave.signer)
        except:
            _wrapException()
            raise

    def stats(self, *args, **kwargs):
        try:
            return self.__slave.stats(*args, **kwargs)
        except:
            _wrapException()
            raise

    def reloadServerKeys(self, *args, **kwargs):
        # deprecated, not used anymore
        return

    def run_shell(self, *args, **kwargs):
        try:
            return ICQSession(self.__slave.run_shell(*args, **kwargs))
        except:
            _wrapException()
            raise

    def __str__(self):
        try:
            return 'Proxy {0}'.format(self.__slave)
        except:
            _wrapException()
            raise

    def __repr__(self):
        try:
            return '<Proxy {0!r}>'.format(self.__slave)
        except:
            _wrapException()
            raise

    def __enter__(self):
        try:
            self.__slave.__enter__()
            return self
        except:
            _wrapException()
            raise

    def __exit__(self, exc_type, exc_val, exc_tb):
        try:
            self.__slave.__exit__(exc_type, exc_val, exc_tb)
        except:
            _wrapException()
            raise


def Client(implementation='cqudp', *args, **kwargs):
    """
    Данная функция представляет собой фабрику клиентов :class:`.ICQClient`. И является единственно
    правильным способом порождения экземпляров :class:`.ICQClient`.

    На момент написания документации существуют одна используемая реализация: ``cqudp``.
    Старые реализации ``cqueue`` и ``kqueue`` в настоящее время отключены и не поддерживаются.

    :rtype: :class:`.ICQClient`
    """
    version = kwargs.pop('version', 'cg')

    klass = _getClientClass(implementation, version=version)
    if (kwargs.get('useGevent') or (implementation == 'cqudp' and useGevent())) and 'select_function' not in kwargs:
        import gevent.select
        kwargs['select_function'] = gevent.select.select
    return ICQClient(klass(*args, **kwargs), implementation, args, kwargs)


def _reconstruct_client(implementation, args, kwargs):
    return Client(implementation, *args, **kwargs)


@memoized
def _getClientClass(implementation, version='cg'):
    name = _getServiceName(implementation)

    # FIXME (torkve) it would be great to have some common mechanism to mark
    # that we need srvmngr-free implementation and description where to import
    # it from
    if name == 'cqudp-thinclient' and version == 'internal':
        from ya.skynet.services.cqudp.daemon_client import DaemonClient
        return DaemonClient

    return _getApi(name, version)


@memoized
def _getApi(name, version=None, kind=None):
    if name == 'cqudp' and getattr(sys, 'is_standalone_binary', False):
        from ya.skynet.services.cqudp.client import CqueueClient
        return CqueueClient

    from api.skycore import ServiceManager

    ns = os.getenv('SKYNET_CQ_NAMESPACE', 'skynet')
    # TODO consider \param version?
    return ServiceManager().get_service_python_api(ns, name, kind=kind)


def _getServiceName(implementation):
    if implementation in ('c', 'cqueue'):
        return 'cqueue'
    elif implementation == 'cqueue-beta':
        return 'cqueue-beta'
    elif implementation == 'cqueue-alpha':
        return 'cqueue-alpha'
    elif implementation in (1, 'kqueue', 'k'):
        return "kqueue"
    elif implementation in (2, 'kqueue2'):
        return "kqueue2"
    elif implementation == 'cqudp':
        return 'cqudp'
    elif implementation in ('cqudp-thinclient', 'ct'):
        return 'cqudp-thinclient'
    else:
        raise RuntimeError('Unknown cqudp implementation {0}'.format(implementation))


class CQWarning(Warning):
    pass


def _check_vulnerable_keys(signer):
    vulnerable_keys = []
    for key in signer:
        if (
            key.has_private()
            and (key.type() == 'ssh-dss' or key.type() == 'ssh-rsa' and key.size() < 2047)
        ):
            vulnerable_keys.append(key)
    if vulnerable_keys:
        message = '\n'.join("   {} ({}) {} {}".format(
            key.type(),
            key.size(),
            ':'.join(x.encode('hex') for x in key.fingerprint()),
            key.description(),
        ) for key in vulnerable_keys)
        message = "WARNING! You have potentially vulnerable keys.\nYou are encouraged to replace them:\n" + message
        warnings.warn(message, category=CQWarning)
