# -*- coding: utf-8 -*-
"""Структуры данных и примитивы синхронизации, работающие поверх cqudp."""

import sys
import six

__all__ = [
    'Full',
    'Empty',
    'Timeout',
    'ICQAsyncResult',
    'ICQQueue',
    'ICQPipe',
]


class Full(Exception):
    """
    Исключение выбрасывается, если попытка положить данные в :class:`очередь <.ICQQueue>`
    не завершается за заданное время.
    """
    pass


class Empty(Exception):
    """
    Исключение выбрасывается, если попытка получить данные из :class:`трубы <.ICQPipe>`
    не завершается за заданное время.
    """
    pass


class Timeout(Exception):
    """
    Исключение выбрасывается, если ожидание результата
    не завершается за указанное время.
    """
    pass


def _wrapException():
    e_class, e_val, e_tb = sys.exc_info()
    if e_class.__name__ in ('Timeout', 'Full', 'Empty'):
        e_new_class = globals()[e_class.__name__]
        e_val.__class__ = e_new_class
        six.reraise(e_new_class, e_val, e_tb)


class ICQAsyncResult(object):
    """Асинхронный результат, который возвращается для ряда задач на удалённой стороне."""

    def __init__(self, slave):
        self.__slave = slave

    def is_set(self):
        """
        Проверка, выставлен ли уже результат.

        :return: boolean
        """
        return self.__slave.is_set()

    def get(self, timeout=None):
        """
        Получение результата.

        :param timeout: время ожидания или None. Для неблокирующего вызова необходимо передать timeout=0.
        """
        try:
            return self.__slave.get(timeout=timeout)
        except:
            _wrapException()
            raise


class ICQQueue(object):
    """
    Очередь, доступная для использования как на клиенте, так и на сервере.
    Никогда не создавайте очередь самостоятельно. Для создания очереди
    смотрите :meth:`.ICQClient.createQueue`.
    В значительной симулирует стандартный python-класс :class:`Queue.Queue`.

    На удалённой стороне все вызовы и свойства класса возвращают асинхронный
    результат вида :class:`.ICQAsyncResult`, на локальной — поведение идентично
    стандартной очереди.
    """
    def __init__(self, slave, remote=False):
        self.__slave = slave
        if remote:
            self.task_done = self.__remote_task_done
            self.join = self.__remote_join
            self.qsize = self.__remote_qsize
            self.full = self.__remote_full
            self.empty = self.__remote_empty
            self.put = self.__remote_put
            self.get = self.__remote_get

    def __remote_task_done(self):
        return ICQAsyncResult(self.__slave.task_done())

    def task_done(self):  # pylint: disable-msg=E0202
        """
        Сообщает, что полученное ранее задание из очереди выполнено.
        Используется потребительскими потоками. Для каждого вызова
        get() последующий вызов task_done() сообщает объекту очереди
        что обработка задания завершена.
        """
        return self.__slave.task_done()

    def __remote_join(self):
        return ICQAsyncResult(self.__slave.join())

    def join(self):  # pylint: disable-msg=E0202
        """
        Блокирование выполнения до тех пор, пока все объекты из очереди
        не будут получены и обработаны.
        """
        return self.__slave.join()

    def __remote_qsize(self):
        return ICQAsyncResult(self.__slave.qsize())

    def qsize(self):  # pylint: disable-msg=E0202
        """Текущий размер очереди"""
        return self.__slave.qsize()

    def __remote_full(self):
        return ICQAsyncResult(self.__slave.full())

    def full(self):  # pylint: disable-msg=E0202
        """Проверка, не является ли очередь заполненной до предела"""
        return self.__slave.full()

    def __remote_empty(self):
        return ICQAsyncResult(self.__slave.empty())

    def empty(self):  # pylint: disable-msg=E0202
        """Проверка, не пуста ли очередь"""
        return self.__slave.empty()

    def __remote_put(self, item, timeout=None):
        return ICQAsyncResult(self.__slave.put(item, timeout=timeout))

    def put(self, item, block=True, timeout=None):  # pylint: disable-msg=E0202
        """
        Добавление элемента в очередь.
        На удалённой стороне параметр `block` отсутствует, вызов всегда неблокирующий.
        При этом если элемент не удастся положить в очередь за `timeout`,
        вернётся исключение :class:`.Full`.
        """
        try:
            return self.__slave.put(item, block=block, timeout=timeout)
        except:
            _wrapException()
            raise

    def __remote_get(self):
        return ICQAsyncResult(self.__slave.get())

    def get(self, block=True, timeout=None):  # pylint: disable-msg=E0202
        """
        Получение элемента из очереди.
        На удалённой стороне параметры `block` и `timeout`
        отсутствуют, вызов всегда неблокирующий.
        """
        try:
            return self.__slave.get(block=block, timeout=timeout)
        except:
            _wrapException()
            raise

    def __repr__(self):
        return self.__slave.__repr__()

    def __str__(self):
        return self.__slave.__str__()

    def __reduce__(self):
        return ICQQueue, (self.__slave, True)


class ICQPipe(object):
    """
    Канал для данных, доступная как на клиенте, так и на удалённой стороне.
    Позволяет клиенту посылать данные потоком одному или нескольким удалённым
    хостам, а удалённым хостам — слать данные клиенту.

    Для создания трубы смотрите :meth:`.ICQClient.create_pipe`.
    """

    def __init__(self, slave, remote=False):
        self.__slave = slave
        if remote:
            self.put = self.__remote_put

    def get(self, block=True, timeout=None):
        """
        Получение входящего сообщения.
        Если вызов неблокирующий или результат не был получен за время таймаута,
        выкидывает исключение :class:`.Empty`

        :param bool block: должен ли вызов быть блокирующим
        :param timeout: максимальное время ожидания результата или :data:`None`
        :return: на удалённой машине возвращается сообщение, на локальной —
                 пара ``(хост, сообщение)``
        """
        try:
            return self.__slave.get(block=block, timeout=timeout)
        except:
            _wrapException()
            raise

    def __remote_put(self, item):
        """Послать сообщение с удалённой машины локальному клиенту."""
        return self.__slave.put(item)

    def put(self, item, hosts):  # pylint: disable-msg=E0202
        """
        Послать сообщение. При посылке сообщения с локального клиента необходимо
        указать список машин, на которые его передать. На удалённой стороне
        параметр hosts отсутствует, все сообщения отправляются клиенту.

        :param item: сообщение, которое необходимо передать на удалённую сторону.
        :param hosts: список машин, которым послать сообщение.
                Имена машин должны быть из того списка, который был
                передан клиенту при запуске задачи. На удалённой стороне
                этот параметр отсутствует.

        """
        return self.__slave.put(hosts, item)

    def __reduce__(self):
        return ICQPipe, (self.__slave, True)

    def _get_event_fd(self):
        return self.__slave._get_event_fd()

    def _is_data_ready(self):
        return self.__slave._is_data_ready()
