# -*- coding: utf-8 -*-

from .exceptions import _wrapException


class GeneratorExceptionWrapper(object):
    def __init__(self, iterable):
        self.iterable = iter(iterable)

    def __iter__(self):
        return self

    def send(self, value):
        try:
            return self.iterable.send(value)
        except:
            _wrapException()
            raise

    def throw(self, *args, **kwargs):
        try:
            return self.iterable.throw(*args, **kwargs)
        except:
            _wrapException()
            raise

    def close(self):
        return self.iterable.close()

    def next(self):
        host, result, err = next(self.iterable)
        return host, result, _wrapException(err, needRaise=False) if err is not None else err

    __next__ = next


class ICQSession(object):
    """
    :class:`.ICQSession` — это экземпляр сессии, порождаемый при удалённом исполнении
    пользовательских объектов и контролирующий весь процесс. Из сессии можно получать
    статус работы вашего объекта, результаты его исполнения и ошибки, произошедшие на
    удалённой стороне.

    Сессия поддерживает Python-семантику ``with``: при выходе из ``with``-блока будет
    автоматически вызван метод :meth:`~.ICQSession.shutdown`, останавливающий выполнение
    всех ещё не завершившихся задач:

    .. code-block:: py

        with session:
            for host, result, err in session.wait(10):
                # Do some work
        # После выхода из блока with задача должна прекратить своё исполнение на всех машинах
    """

    def __init__(self, slave):
        self.__slave = slave

    @property
    def running(self):
        """
        Признак того, что эта сессия запущена и продолжает исполнение удалённой задачи.

        Сессия начинает свое исполнение сразу после удачного конструирования c помощью
        :meth:`.ICQClient.run` или его аналогами.

        Сессия завершается в следующих случаях:
          1. Поступление финальных результатов от всех удалённых узлов, участвующих в сессии
          2. Явный вызов :meth:`~.ICQSession.shutdown` или неявный :meth:`~object.__exit__`.
          3. Явный вызов :meth:`~.ICQClient.shutdown` или неявный :meth:`~object.__exit__` у владеющего сессией клиента :class:`~api.cqueue.client.ICQClient`

        :rtype: bool
        """
        try:
            return bool(self.__slave.running)
        except:
            _wrapException()
            raise

    @property
    def id(self):
        """
        Глобальный уникальный идентификатор этой сессии.

        :rtype: str
        """
        try:
            return str(self.__slave.id)
        except:
            _wrapException()
            raise

    def remoteObject(self):
        """
        Объект который вы передали для исполнения на удалённой стороне.

        :rtype: :class:`~.remoteobject.IRemoteObject`
        """
        try:
            return self.__slave.remoteObject
        except:
            _wrapException()
            raise

    def wait(self, timeout=None):
        """
        Этот метод используется для получения статуса исполнения вашего объекта.
        После вызова этого объекта будет порождён генератор результатов, который
        будет существовать в течение времени, указанного в таймауте.

        Генератор будет выбрасывать тройки:
            * имя удалённого узла,
            * результат работы,
            * исключение, произошедшее на удалённом узле

        .. NOTE::

            Данный метод фактически возвращает генератор, по которому можно пройтись
            оператором `for` или вызывать его методы `.next()` для итерации.
            Когда вы пользуетесь методом :meth:`~.ICQClient.iter`, то вы фактически
            получаете генератор, который проходится по массиву генераторов
            (по одному для каждой машины). В этом случае, чтобы обозначить конец
            результатов для какой-то машины, очередной шаг генератора вернёт тройку
            (``host``, :data:`None`, :exc:`~exceptions.StopIteration`), учитывайте это в своих
            скриптах.

        Именем удалённого узла является имя, переданное пользователем (dns, ip, etc).

        В случае одновременного порождения сразу нескольких генераторов методом
        :meth:`~.ICQSession.wait` и попытке итерарирования по ним любыми способами,
        результаты работы будут поступать в случайный из порождённых на данный
        момент генераторов, но один и тот же результат поступит только в один из них.

        :param timeout: Служит для задания времени жизни генератора. По истечении
                        заданного таймаута генератор будет закрыт выкидыванием
                        исключения :exc:`~exceptions.StopIteration`. В случае закрытия
                        генератора, пользователь может узнать, продолжает ли сессия своё
                        исполнение с помощью свойства :attr:`~.ICQSession.running`. Если
                        сессия продолжает исполнение, пользователь может позвать новый
                        :meth:`~.ICQSession.wait` с таймаутом. Если же сессия завершена
                        и в предыдущий раз был вызван :meth:`~.ICQSession.wait` c
                        таймаутом, или же, если результаты из сессии не забирались вовсе,
                        желательно прочитать все возможно оставшиеся результаты вызовом
                        метода :meth:`~.ICQSession.wait` без указани таймаута.
        :type timeout: int or long or float or None
        :rtype: tuple[str, object or None, object or None]

        """
        try:
            return GeneratorExceptionWrapper(self.__slave.wait(timeout))
        except:
            _wrapException()
            raise

    def poll(self, timeout=None):
        """
        То же, что и :meth:`~.ICQSession.wait` с той лишь разницей, что генератор
        закрывается сразу после получения первых результатов, не обязательно
        исчерпывая весь отведённый таймаут.

        Поведение :meth:`~.ICQSession.poll` и :meth:`~.ICQSession.wait` в случае
        ``timeout=0`` абсолютно идентично: вызов не блокирует выполнение, возвращая
        только уже пришедшие результаты.
        """
        try:
            return GeneratorExceptionWrapper(self.__slave.poll(timeout))
        except:
            _wrapException()
            raise

    def link(self, receiver=None):
        raise NotImplementedError("not supported anymore")

    @property
    def participantsFoundHandler(self):
        pass

    @participantsFoundHandler.setter
    def participantsFoundHandler(self, receiver):
        raise NotImplementedError("not supported anymore")

    @participantsFoundHandler.deleter
    def participantsFoundHandler(self):
        raise NotImplementedError("not supported anymore")

    @property
    def resultEventsHandler(self):
        pass

    @resultEventsHandler.setter
    def resultEventsHandler(self, receiver):
        raise NotImplementedError("not supported anymore")

    @resultEventsHandler.deleter
    def resultEventsHandler(self):
        raise NotImplementedError("not supported anymore")

    def shutdown(self):
        """
        Немедленно прекращает исполнение этой сессии. Если при этом продолжается
        исполнение пользовательских объектов на удалённых узлах, это исполнение
        аварийно завершается. В случае, если пользователь по каким-либо причинам
        больше не нуждается в результатах работы удалённых объектов, рекомендуется
        как можно раньше позвать этот метод. Например, если был вызван метод
        :meth:`~.ICQSession.wait` с таймаутом и таймаут истек и больше не
        предполагается получать результаты из этой сессии, разумно позвать этот
        метод, чтобы освободить ресурсы удалённых узлов от исполнения работы,
        результаты которой больше не нужны.

        :meth:`~.ICQSession.shutdown` автоматически будет вызван при выходе из
        блока ``with`` или при сборе сессии сборщиком мусора.
        """
        try:
            self.__slave.shutdown()
        except:
            _wrapException()
            raise

    def _get_event_fd(self):
        return self.__slave._get_event_fd()

    def _is_data_ready(self):
        return self.__slave._is_data_ready()

    def __str__(self):
        try:
            return 'I {0}'.format(self.__slave)
        except:
            _wrapException()
            raise

    def __repr__(self):
        try:
            return '<I {0!r}>'.format(self.__slave)
        except:
            _wrapException()
            raise

    def __enter__(self):
        try:
            self.__slave.__enter__()
            return self
        except:
            _wrapException()
            raise

    def __exit__(self, exc_type, exc_val, exc_tb):
        try:
            self.__slave.__exit__(exc_type, exc_val, exc_tb)
        except:
            _wrapException()
            raise

    def __reduce__(self):
        if self.running:
            raise ValueError("Cannot pickle running cqudp session")
        return bool, ()  # there's no easy way to serialize myself as None
