# -*- coding: utf-8 -*-

__author__ = 'kaliki'


class ICQSigner(object):
    """
    Интерфейс менеджера ключей. Позволяет управлять ключами клиента, добавляя
    или удаляя их по своему желанию, а также подписывать произвольные данные.
    """

    def __init__(self, slave):
        self.__slave = slave

    def __iter__(self):
        """
        Итерация по всем ключам, зарегистрированным на данный момент в клиенте.
        Данный генератор возвращает объекты типа :class:`.ICQKey`.
        """
        for key in self.__slave:
            yield ICQKey(key)

    def hash(self, data=''):
        """
        Взятие хэша от произвольных данных.
        Возвращает строку с хэшем в бинарном виде.

        :param bytes data: данные для хэширования
        :rtype: bytes
        """
        return self.__slave.hash(data).digest()

    def sign(self, dataHash, fingerprints=None):
        """
        Подписание хэша, полученного методом :meth:`~.ICQSigner.hash`.

        :param bytes dataHash: хэш, который требуется подписать
        :param fingerprints: список отпечатков ключей. Если указан,
            хэш будет подписан только этими ключами
        :type fingerprints: iterable[bytes]
        :return: генератор, возвращающий подписи поштучно
        """
        for signature in self.__slave.sign(dataHash, fingerprints):
            yield signature

    def add_key(self, key):
        """
        Зарегистрировать ключ.

        .. versionchanged:: 17.0.0
            Старое название метода: addKey

        :param key: ключ, который требуется зарегистрировать в менеджере ключей
        :type key: :class:`.ICQKey`
        """
        return self.__slave.addKey(key._ICQKey__slave)

    addKey = add_key

    def remove_key(self, key):
        """
        Удалить ключ из менеджера.

        .. versionchanged:: 17.0.0
            Старое название метода: removeKey

        :param key: ключ, который требуется удалить.
        :type key: :class:`.ICQKey`
        """
        return self.__slave.removeKey(key._ICQKey__slave)

    removeKey = remove_key

    def load_keys(self, stream, comment=''):
        """
        Прочитать ключи из потока данных.

        .. NOTE::

            Данные ключи не будут автоматически зарегистрированы в менеджере.

        .. versionchanged:: 17.0.0
            Старое название метода: loadKeys

        :param stream: произвольный поток, поддерживающий чтение из него данных
        :param str comment: опциональный комментарий, который будет добавлен к
            каждому прочитанному ключу
        :return: генератор, возвращающий прочитанные из потока ключи поштучно
        """
        for key in self.__slave.loadKeys(stream, comment):
            yield ICQKey(key)

    loadKeys = load_keys

    def loads_keys(self, data, comment=''):
        """
        Прочитать ключи из строки.

        .. NOTE::

            Данные ключи не будут автоматически зарегистрированы в менеджере.

        .. versionchanged:: 17.0.0
            Старое название метода: loadsKeys

        :param bytes data: строка с ключами
        :param str comment: опциональный комментарий, который будет добавлен к
            каждому прочитанному ключу
        :return: генератор, возвращающий прочитанные из строки ключи поштучно
        """
        for key in self.__slave.loadsKeys(data, comment):
            yield ICQKey(key)

    loadsKeys = loads_keys

    def generate_key(self, bits=2048, key_type='rsa'):
        """
        Сгенерировать новый ключ.

        .. NOTE::

            Данный ключ не будет автоматически зарегистрирован в менеджере.

        .. versionchanged:: 17.0.0
            Старое название метода: generateKey

        :param int bits: желаемая длина ключа в битах
        :param key_type: тип ключа. Поддерживаются следующие типы:

            - ``rsa`` — RSA_-ключ, разрешённая длина от 1024 до 2048 бит
            - ``dsa`` — DSA_-ключ, разрешённая длина от 512 до 1024 бит
            - ``ecdsa`` - ECDSA_-ключ, разрешённые длины: 256, 384 или 521 бит

        :return: сгенерированный ключ
        :rtype: :class:`.ICQKey`

        .. _RSA: http://en.wikipedia.org/wiki/RSA_%28algorithm%29
        .. _DSA: http://en.wikipedia.org/wiki/Digital_Signature_Algorithm
        .. _ECDSA: https://en.wikipedia.org/wiki/Elliptic_Curve_Digital_Signature_Algorithm
        """
        return ICQKey(self.__slave.generateKey(bits, key_type))

    def generateKey(self, bits=2048, keyType='rsa'):
        return self.generate_key(bits=bits, key_type=keyType)

    def __str__(self):
        return 'Proxy {0}'.format(self.__slave)

    def __repr__(self):
        return '<Proxy {0!r}>'.format(self.__slave)


class ICQKey(object):
    """
    Интерфейс стандартного ключа cqudp.
    """

    def __init__(self, slave):
        self.__slave = slave

    def type(self):
        """
        Тип ключа. Возвращает сигнатуру, которой ключ этого типа помечается
        в ssh (ssh-rsa, ssh-dss, ecdsa-sha2-nistp384 и т.п.)

        :rtype: str
        """
        return self.__slave.type

    def size(self):
        """
        Длина ключа в битах

        :rtype: int
        """
        return self.__slave.size()

    def description(self):
        """
        Описание ключа.
        Содержит комментарий из файла с ключом или ssh-агента

        :rtype: str
        """
        return self.__slave.description()

    def has_private(self):
        """
        Проверка, имеет ли ключ приватную часть для подписи

        :rtype: bool
        """
        return self.__slave.has_private()

    hasPrivate = has_private

    def public(self):
        """
        Публичная часть ключа.

        Публичный ключ не позволяет подписывать данные, но позволяет верифицировать
        подпись, сделанную приватным ключом.

        :rtype: :class:`.ICQKey`
        """
        return ICQKey(self.__slave.public())

    def fingerprint(self):
        """
        Отпечаток ключа в бинарном виде
        """
        return self.__slave.fingerprint()

    def export(self):
        """
        Экспортировать ключ в виде ASCII-строки.
        """
        return self.__slave.export()

    def __str__(self):
        return 'Proxy {0}'.format(self.__slave)

    def __repr__(self):
        return '<Proxy {0!r}>'.format(self.__slave)
