import sys

try:
    __import__('pkg_resources').require('requests')
    import requests
    requests.packages.urllib3.disable_warnings()
except BaseException:
    if getattr(sys, 'is_standalone_binary', False):
        raise

from collections import defaultdict
from multiprocessing import Pool, cpu_count
from itertools import chain


class GenCfgHttpError(Exception):
    def __init__(self, code, reason):
        self.code = code
        self.reason = reason
        if code:
            super(GenCfgHttpError, self).__init__('GenCfg api http error: {0} {1}'.format(code, reason))
        else:
            super(GenCfgHttpError, self).__init__('GenCfg api http error: {0}'.format(reason))


class GenCfgNotFoundError(Exception):
    def __init__(self, url):
        super(GenCfgNotFoundError, self).__init__("GenCfg '{}' not found".format(url))


class GenCfgBadRequest(Exception):
    def __init__(self, url, err_msg):
        super(GenCfgBadRequest, self).__init__("GenCfg '{}' failed with: {}".format(url, err_msg))


def get_mtns_global(args):
    self, group_name, tag, instance_tags = args
    return self.get_mtns(group_name, tag, instance_tags)


class GenCfg(object):
    MAX_PARALLEL_REQUESTS = 16
    API_URL_TRUNK_INSTANCES_DETAILED = 'http://api.gencfg.yandex-team.ru/trunk/searcherlookup/groups/{}/instances'
    API_URL_TAG_INSTANCES_DETAILED = 'http://api.gencfg.yandex-team.ru/tags/{}/searcherlookup/groups/{}/instances'
    API_URL_TRUNK_INSTANCES = 'http://api.gencfg.yandex-team.ru/trunk/groups/{}/instances'
    API_URL_TAG_INSTANCES = 'http://api.gencfg.yandex-team.ru/tags/{}/groups/{}/instances'
    API_URL_ONLINE_INSTANCES = 'http://api.gencfg.yandex-team.ru/online/groups/{}/instances'
    API_URL_TRUNK_HOSTS = 'http://api.gencfg.yandex-team.ru/trunk/groups/{}'
    API_URL_TAG_HOSTS = 'http://api.gencfg.yandex-team.ru/tags/{}/groups/{}'

    def __init__(self):
        self.timeouts = [15, 20, 25]

    def get_hosts(self, group_name, tag=None, instance_tags=None):
        # process ONLINE
        if tag and tag.lower() == 'online':
            url = self.API_URL_ONLINE_INSTANCES.format(group_name)
            return [instance['hostname'] for instance in self._do_request(url)['instances']]

        # process instance_tag filter
        if instance_tags:
            if tag:
                url = self.API_URL_TAG_INSTANCES_DETAILED.format(tag, group_name)
            else:
                url = self.API_URL_TRUNK_INSTANCES_DETAILED.format(group_name)

            return [instance['hostname'] for instance in self._do_request(url)['instances']
                    if not(instance_tags - set(instance['tags']))]

        # standard host resolving
        if tag:
            url = self.API_URL_TAG_HOSTS.format(tag, group_name)
        else:
            url = self.API_URL_TRUNK_HOSTS.format(group_name)

        return self._do_request(url)['hosts']

    def get_instances(self, group_name, tag=None, cut_tag=False, hosts=None, instance_tags=None):
        if tag:
            url = self.API_URL_ONLINE_INSTANCES.format(group_name) if tag.lower() == 'online' else \
                self.API_URL_TAG_INSTANCES.format(tag, group_name)
        else:
            url = self.API_URL_TRUNK_INSTANCES.format(group_name)

        result = defaultdict(set)
        for instance in self._do_request(url)['instances']:
            hostname = instance['hostname']
            if hosts and hostname not in hosts:
                # skip instances for not requested hosts
                continue
            if instance_tags and instance_tags - set(instance["tags"]):
                continue
            shard = instance.get('shard_name', 'none')
            if cut_tag:
                result[hostname].add((shard, '{}:{}'.format(hostname, instance['port'])))
            else:
                result[hostname].add((shard, '{}:{}@{}'.format(hostname, instance['port'], tag if tag else 'trunk')))

        return result

    def get_slave_mtns(self, group_name, tag=None, instance_tags=None):
        if tag:
            url = self.API_URL_TAG_HOSTS.format(tag, group_name)
        else:
            url = self.API_URL_TRUNK_HOSTS.format(group_name)

        url += '/card'
        slave_groups = self._do_request(url)['slaves']

        size = len(slave_groups)
        slave_instances = Pool(min(cpu_count(), self.MAX_PARALLEL_REQUESTS)).map(
            get_mtns_global,
            zip([self] * size, slave_groups, [tag] * size, [instance_tags] * size)
        )

        result = list(chain.from_iterable(slave_instances))
        return result

    def get_mtns(self, group_name, tag=None, instance_tags=None):
        if tag:
            url = self.API_URL_ONLINE_INSTANCES.format(group_name) if tag.lower() == 'online' else \
                self.API_URL_TAG_INSTANCES_DETAILED.format(tag, group_name)
        else:
            url = self.API_URL_TRUNK_INSTANCES_DETAILED.format(group_name)

        result = []
        for instance in self._do_request(url)['instances']:
            try:
                if not instance_tags or not(instance_tags - set(instance['tags'])):
                    hostname = instance['hbf']['interfaces']['backbone']['hostname']
                    result.append(hostname)
            except BaseException:
                # ignore instances without HBF
                pass
        return result

    def _do_request(self, url):
        exc = None
        for timeout in self.timeouts:
            try:
                r = requests.get(url, timeout=timeout)
                exc = None
                break
            except requests.exceptions.Timeout:
                exc = 'timeout'
            except requests.exceptions.ConnectionError as e:
                exc = str(e)

        if exc is not None:
            raise GenCfgHttpError(0, exc)

        if r.status_code != requests.codes.ok:
            if r.status_code == requests.codes.not_found:
                raise GenCfgNotFoundError(url)
            if r.status_code == requests.codes.bad_request:
                raise GenCfgBadRequest(url, r.json()['error'])
            # r.reason was not available in this version of requests, that's why
            raise GenCfgHttpError(r.status_code, "")

        return r.json()
