import os
from kernel.util.functional import memoized, threadsafe


@threadsafe
@memoized
def heartbeat_client(namespace=None):
    if namespace is None:
        namespace = os.getenv('SKYNET_HEARTBEAT_NAMESPACE', 'skynet')

    from api.skycore import ServiceManager

    return ServiceManager().get_service_python_api(namespace, 'heartbeat-client')()


def schedule_report(
        name,
        report,
        valid=3600,
        compress=True,
        incremental=False,
        discards=None,
        sendDelay=900
):
    """
    Schedule a report given to be sent to heartbeat-server instance in a regular way.
    :param name:        Report type (name).
    :param report:      Report data itself.
    :param valid:       Amount of seconds the report will be valid.
    :param compress:    Flags perform bzip2 compression or not.
    :param incremental: Flags should the report sent if the scheduler has another report of the same type
                        in the not yet sent reports queue.
    :param discards:    A list of report type names, which this report additionally discards.
                        I.e., during the report scheduling, all scheduled reports, which are listed here,
                        will be discarded.
    :param sendDelay:   Amount of seconds the report sending should be postponed on.
    :return:            None
    """
    return heartbeat_client().call(
        'scheduleReport',
        name,
        report,
        valid=valid,
        compress=compress,
        incremental=incremental,
        discards=discards,
        sendDelay=sendDelay
    )


def schedule_report_async(
        name,
        report,
        valid=3600,
        compress=True,
        incremental=False,
        discards=None,
        sendDelay=900
):
    """
    Schedule a report given to be sent to heartbeat-server instance in an async way.
    Put it to a queue instead of direct database writing.
    NOTE: reports can be lost while client restarts.

    :param name:        Report type (name).
    :param report:      Report data itself.
    :param valid:       Amount of seconds the report will be valid.
    :param compress:    Flags perform bzip2 compression or not.
    :param incremental: Flags should the report sent if the scheduler has another report of the same type
                        in the not yet sent reports queue.
    :param discards:    A list of report type names, which this report additionally discards.
                        I.e., during the report scheduling, all scheduled reports, which are listed here,
                        will be discarded.
    :param sendDelay:   Amount of seconds the report sending should be postponed on.
    :param asyncMode:   False - write report directly to database,
                        True - put it in a queue (NOTE: reports can be lost while server restarts).
    :return:            None
    """
    return heartbeat_client().call(
        'scheduleReportAsync',
        name,
        report,
        valid=valid,
        compress=compress,
        incremental=incremental,
        discards=discards,
        sendDelay=sendDelay
    )


def get_plugins():
    """
    Provides a list of all known heartbeat-client plugins.
    :return:            List of plugin names
    """
    return heartbeat_client().call('get_plugins')


def get_last_report(plugin_name):
    """
    Returns a last generated report for the specified plugin.
    :param name:        Name of a plugin.
    :return:            Last plugin's report (usually as a dict)
    """
    return heartbeat_client().call('get_last_report', plugin_name)


scheduleReport = schedule_report
scheduleReportAsync = schedule_report_async
