from __future__ import print_function

import os
import sys
import time
import warnings
from functools import partial

from kernel.util.functional import memoized, threadsafe
from kernel.util import logging


daemonName = sys.argv[0]


@threadsafe
@memoized
def get_root():
    from api.srvmngr import getRoot
    return getRoot()


@threadsafe
@memoized
def LoggerClass():
    from api.skycore import ServiceManager
    return ServiceManager().get_service_python_api('skynet', 'logger')


@memoized
def Logger():
    return LoggerClass()()


def _format(app, level, source, message):
    ct = time.time()
    if sys.version_info[0] == 3:
        msecs = (ct - int(ct)) * 1000
    else:
        msecs = (ct - long(ct)) * 1000  # noqa
    heading = '%s.%03d %2s [%s%s]  ' % (
        time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(ct)),
        msecs, level, app, source
    )

    result = []
    for idx, line in enumerate(message.strip().split('\n')):
        if idx == 0:
            result.append(' '.join((heading, line)))
        else:
            result.append(' '.join((' ' * len(heading), line.rstrip())))

    return '\n'.join(result)


def raw(level, message, source=None, app='unknown', filename=None):
    """
    raw logging function - use it if you need non existent error level.
    sends :arg str message: with :arg int level: and :arg str app: to
    logger service, that will write it into :arg str filename:.
    optionally duplicates message to stdout if SKYNET_LOG_STDOUT is set.
    if it failes to send message - tries to write to 'fallback_logfile'.
    """
    source = source or daemonName
    src = ':%s' % source if source != 'root' else ''
    try:
        if os.getenv('SKYNET_LOG_STDOUT', False) == '1':
            print(_format(app, level, src, message))
        Logger().log(app, source, level, message, filename)
    except Exception as ex:
        # fallback mode if we failed to log exception
        # bad: file path is copy-pasted from server implementation
        # in 99% of cases error was "no buffer space available", which
        # means udp sending failed.
        if filename is None:
            filename = 'logfile'

        if isinstance(level, int):
            if level == 0:
                level = '[----]'
            elif level == 9:
                level = '[PROT]'
            elif level == 10:
                level = '[DEBG]'
            elif level == 20:
                level = '[INFO]'
            elif level == 30:
                level = '[WARN]'
            elif level == 40:
                level = '[ERRR]'
            elif level == 50:
                level = '[CRIT]'

        try:
            root = get_root()
        except Exception:
            # we know nothing about skynet so cannot log directly
            return

        try:
            try:
                formatted = _format(app, level, src, message)
            except:
                formatted = '{0}:{1}:{2}'.format(app, level, message)

            fd = os.open(os.path.join(root, 'var', 'log', filename), os.O_APPEND | os.O_WRONLY)
            try:
                logfile = os.fdopen(fd, 'a')
            except:
                os.close(fd)

            try:
                logfile.write(formatted + '  [directly logged, because of "' + str(ex) + '"] \n')
            finally:
                logfile.close()
        except EnvironmentError:
            pass

logLevelHelperFuncs = {}  # noqa


def installLogLevelHelpers():
    '''
    Create shortcuts for simple usage.

    .. rubric:: Example

    >>> from api import logger as log
    >>> log.debug('Some debug stuff')
    >>>
    >>> from api import logger
    >>> log = logger.makeLogger(source='my facility')
    >>> log.debug('Some debug stuff')
    '''
    logLevels = dict(
        dump=-2,    # Use only for dumping HUGE amount of data. Don't use it until log filtering is implemented
        debug=-1,   # Detailed execution flow description
        normal=0,   # Execution flow milestones
        warning=1,  # Not a problem. For example, unexpected socket read timed out
        minor=2,    # Minor problem
        major=3,    # Serious problem
        critical=4  # Critical problem, program WILL exit now (maybe after completing its cleanup).
    )

    for level in logLevels:
        logger = partial(raw, logLevels[level])
        logLevelHelperFuncs[level] = logger
        globals()[level] = logger

installLogLevelHelpers()  # noqa


def makeLogger(**kwargs):
    '''
    Allows wrapping all log funcs to some source.

    This two are identical:
     - makeLogger(source='some src').debug('asd')
     - debug('asd', source='some src')

    See tests.py:test_log_wrapper() for more details.
    '''
    warnings.warn(
        'makeLogger is deprecated, use constructLogger(app=XXX) instead, '
        'will be removed soon',
        DeprecationWarning
    )

    class _wrappedLogger(object):
        def __init__(self):
            for loggerName, logger in logLevelHelperFuncs.items():
                setattr(self, loggerName, partial(logger, **kwargs))

    return _wrappedLogger()


class SkynetLoggingHandler(logging.Handler):
    """
    Handler which can be used with python logging daemon.

    .. rubric:: Example

    >>> from api.logger import SkynetLoggingHandler
    >>> import logging
    >>> skynetLogHandler = SkynetLoggingHandler()
    >>> log = logging.getLogger('skynet.myapp')
    >>> log.setLevel(logging.DEBUG)
    >>> log.addHandler(skynetLogHandler)
    >>> log.info('Hi there!')

    The above example will produce this log line::

       Apr 11 16:46:25 [skynet.myapp] INFO   : Hi there!

    (format will likely change in the future)
    """

    appName = None
    __DefaultValue__ = object()

    def __init__(self, app=__DefaultValue__, filename=None, *args, **kwargs):
        if app is self.__DefaultValue__:
            warnings.warn(
                'Usage of SkynetLoggingHandler without app= is deprecated. '
                'Use app= instead of source=',
                DeprecationWarning
            )
            app = 'unknown'
        self.appName = app
        self.filename = filename
        logging.Handler.__init__(self, *args, **kwargs)

    def emit(self, record):
        raw(record.levelno, self.format(record), record.name, self.appName, filename=self.filename)

    def close(self):
        logging.Handler.close(self)
        if isinstance(Logger, memoized):
            if Logger.cache:
                Logger().close()
                Logger.cache.clear()


def constructLogger(source='', app='unknown', filename=None):
    log = logging.getLogger(source)

    log.setLevel(logging.DEBUG)

    handler = SkynetLoggingHandler(app=app, filename=filename)
    log.addHandler(handler)
    log.normal = log.info

    return log
