import sys

try:
    __import__('pkg_resources').require('requests')
    import requests
    requests.packages.urllib3.disable_warnings()
except BaseException:
    if getattr(sys, 'is_standalone_binary', False):
        raise

try:
    import simplejson as json
except ImportError:
    import json


class NetmonHttpError(Exception):
    def __init__(self, code, reason):
        self.code = code
        self.reason = reason
        if code:
            super(NetmonHttpError, self).__init__('Netmon api http error: {0} {1}'.format(code, reason))
        else:
            super(NetmonHttpError, self).__init__('Netmon api http error: {0}'.format(reason))


class NetmonResolver(object):
    NETMON_API_URL = 'https://netmon-resolver.in.yandex-team.ru/api/expression/v1/expand'

    def __init__(self):
        self.timeouts = [5, 10, 15]

    def get_hosts_by_dc(self, dc):
        value, mode = self._parse_value(dc)
        request = {'expression': 'dc={}{}'.format(value, mode)}
        return self._do_request(request)

    def get_hosts_by_line(self, line):
        value, mode = self._parse_value(line)
        request = {'expression': 'queue={}{}'.format(value, mode)}
        return self._do_request(request)

    @staticmethod
    def _parse_value(value):
        chunks = value.split(':')
        value = chunks[0]
        mode = chunks[1] if len(chunks) > 1 else 'all'
        mode_str = {
            'all': '',
            'hw': ' - virtual=true',
            'virtual': ' - virtual=false'
        }.get(mode, '')

        return value, mode_str

    def _do_request(self, request):
        exc = None
        for timeout in self.timeouts:
            try:
                r = requests.post(self.NETMON_API_URL, json.dumps(request), timeout=timeout)
                exc = None
                break
            except requests.exceptions.Timeout:
                exc = 'timeout'
            except requests.exceptions.ConnectionError as e:
                exc = str(e)

        if exc is not None:
            raise NetmonHttpError(0, exc)

        if r.status_code != requests.codes.ok:
            # r.reason was not available in this version of requests, that's why
            raise NetmonHttpError(r.status_code, "")

        return r.json()['hosts']
