from library.poll import SimpleTaskPoll, CopyTaskPoll


class IPoller(object):
    """
    Poll skynet tasks. Suitable for most of the tasks, except :py:class:`api.copier.ICopier` ones
    (however, :meth:`api.copier.ICopier.copy`'ed ones are supported).

    .. rubric:: Example

    >>> from api.poll import Poller
    >>> poll = Poller()
    >>> poll.add(task1)
    >>> poll.add(task2)
    >>> while not poll.finished():
    ...     finishedTasks = poll.poll(0.1) # 100 ms
    ...     for task in finishedTasks:
    ...         if task.reason is not None:
    ...             print task.reason

    In the example above we create a loop polling for the 100ms and outputing the errors.
    You can use :meth:`.IPoller.poll` without argument, forcing it block until at least
    one of the tasks is finished:

    >>> while not poll.finished():
    ...    finishedTasks = poll.poll()
    ...     for task in finishedTasks:
    ...         if task.reason is not None:
    ...             print task.reason
    """

    def __init__(self, slave, tasks=None):
        self.__slave = slave(tasks)

    def add(self, task):
        """
        Add the task to the list of polled ones.

        :arg unknown task: Task to poll
        """
        return self.__slave.add(task)

    def remove(self, task):
        """
        Remove the task from the list of polled ones.
        You should remember, that finished ones are removed automatically
        after poll, so that they don't require removing.

        :arg unknown task: Task to remove
        """
        return self.__slave.remove(task)

    def finished(self):
        """
        Check if the poll is finished.
        Returns if all the tasks polled are finishedTasks

        :rtype: bool
        """
        return self.__slave.finished()

    def poll(self, timeout=None):
        """
        Poll for the tasks.
        If timeout specified, it polls for that time, otherwise
        blocks until at least one task is finished.
        Returns list of the tasks finished.

        :arg timeout: timeout in seconds
        :type timeout: float or int or None

        :rtype: list
        """
        return self.__slave.poll(timeout)


def Poller(tasks=None, implementation='kqueue'):
    """
    Create the poll object

    :arg tasks: Initial list of tasks to poll
    :type tasks: list or None

    :arg implementation: 'kqueue' or 'scp'

    :rtype: :py:class:`api.poll.IPoller`
    """
    if implementation == 'kqueue':
        impl = SimpleTaskPoll
    elif implementation == 'scp':
        impl = CopyTaskPoll
    else:
        raise ValueError("Unknown implementation")

    return IPoller(impl, tasks)
