import sys

try:
    from dns import resolver
except ImportError:
    if getattr(sys, 'is_standalone_binary', False):
        raise
    resolver = None

try:
    __import__('pkg_resources').require('requests')
    import requests
    requests.packages.urllib3.disable_warnings()
except BaseException:
    if getattr(sys, 'is_standalone_binary', False):
        raise


class QloudHttpError(Exception):
    def __init__(self, code, reason):
        self.code = code
        self.reason = reason
        if code:
            super(QloudHttpError, self).__init__('Qloud API HTTP error: {0} {1}'.format(code, reason))
        else:
            super(QloudHttpError, self).__init__('Qloud API HTTP error: {0}'.format(reason))


class QloudUnknownInstallationError(Exception):
    def __init__(self, installation):
        super(QloudUnknownInstallationError, self).__init__('Unknown Qloud installation: {}'.format(installation))


class QloudDNS(object):
    def __init__(self):
        self.resolver = resolver.Resolver() if resolver else None

    def resolve_hosts(self, fqdn):
        return self._resolve('_host_.'+fqdn)

    def resolve_ids(self, fqdn):
        return self._resolve('_id_.'+fqdn)

    def resolve_ips(self, fqdn):
        return self._resolve('_ip_.'+fqdn)

    def _resolve(self, fqdn):
        result = []
        if self.resolver:
            answer = resolver.query(fqdn, 'SRV')
            for record in answer:
                result.append(str(record.target)[0:-1])

        return result


class QloudHostResolver(object):
    QLOUD_HOSTS_BASE_URL = "https://{}.yandex-team.ru/api/v1/hosts/fqdns"
    QLOUD_HW_SEGMENT_QUERY = 'hardwareSegment={}'
    QLOUD_STATE_QUERY = 'state={}'
    QLOUD_DC_QUERY = 'datacenter={}'
    QLOUD_INSTALLATIONS = {
        'test': 'qloud-test',
        'main': 'qloud',
        'ext': 'qloud-ext'
    }

    def __init__(self):
        self.timeouts = [5, 10, 15]

    def resolve_hosts(self, installation, hardware_segment, state, dc):
        delimiter = '?'
        query = ''

        if hardware_segment:
            query += delimiter + self.QLOUD_HW_SEGMENT_QUERY.format(hardware_segment)
            delimiter = '&'

        if state:
            query += delimiter + self.QLOUD_STATE_QUERY.format(state)
            delimiter = '&'

        if dc:
            query += delimiter + self.QLOUD_DC_QUERY.format(dc)

        try:
            url = self.QLOUD_HOSTS_BASE_URL.format(self.QLOUD_INSTALLATIONS[installation]) + query
        except KeyError:
            raise QloudUnknownInstallationError(installation)

        return self._do_request(url)

    def _do_request(self, url):
        exc = None
        for timeout in self.timeouts:
            try:
                r = requests.get(url, timeout=timeout, verify=False)
                exc = None
                break
            except requests.exceptions.Timeout:
                exc = 'timeout'
            except requests.exceptions.ConnectionError as e:
                exc = str(e)

        if exc is not None:
            raise QloudHttpError(0, exc)

        if r.status_code != requests.codes.ok:
            # r.reason was not available in this version of requests, that's why
            raise QloudHttpError(r.status_code, "")

        return r.json()
