import sys

try:
    __import__('pkg_resources').require('requests')
    import requests
    requests.packages.urllib3.disable_warnings()
except BaseException:
    if getattr(sys, 'is_standalone_binary', False):
        raise


class SamogonException(Exception):
    pass


class SamogonError(Exception):
    def __init__(self, description):
        super(SamogonError, self).__init__('Samogon error: {}'.format(description))


class SamogonHttpError(SamogonException):
    def __init__(self, code=0, reason='', url=''):
        self.code = code
        self.reason = reason
        if code:
            super(SamogonHttpError, self).__init__('Samogon API error({}): {} {}'.format(url, code, reason))
        else:
            super(SamogonHttpError, self).__init__('Samogon API error({}): {}'.format(url, reason))


# just for backward compatibility
SamogonTimeoutError = SamogonHttpError


class SamogonNotFoundError(SamogonException):
    def __init__(self, service='', error_msg=''):
        super(SamogonNotFoundError, self).__init__('Samogon service {} not found: {}'.format(service, error_msg))


class SamogonResolver(object):
    API_URL_HOSTS = 'https://clusterapi-{}.{}.yandex-team.ru/specific/skylist'
    API_URL_HOST_LIST = 'https://{}.{}.yandex-team.ru/host_list'
    API_SERVANT_QUERY = 'servant={}'
    API_LOCATION_QUERY = 'location={}'

    def __init__(self):
        self.timeouts = [10, 20, 30]

    def get_hosts(self, service, servant=None, location=None):
        service = service.lower()
        servant = servant.lower() if servant else None
        location = location.upper() if location else None

        return self._get_hosts(service, servant, location)

    def _get_hosts(self, service, servant, location):
        delimiter = '?'
        query = ''
        urls = [(self.API_URL_HOSTS, self._process_hosts)]

        if servant:
            query += delimiter + self.API_SERVANT_QUERY.format(servant)
            delimiter = '&'

        if location:
            query += delimiter + self.API_LOCATION_QUERY.format(location)
        else:
            # location is not supported by old URL
            urls.append((self.API_URL_HOST_LIST, self._process_host_list))

        err_msg = ''
        for url, result_processor in urls:
            for source in ['n', 'qloud']:
                try:
                    result = self._do_request(url.format(service, source) + query)
                    if result_processor:
                        result = result_processor(result)

                    return result

                except SamogonException as ex:
                    err_msg += '\n' + ex.message
                    continue

        raise SamogonNotFoundError(service, err_msg)

    @staticmethod
    def _process_hosts(result):
        if isinstance(result, dict):
            if 'value' in result:
                return result['value']

            if 'error' in result:
                raise SamogonError(result['error'])

            raise SamogonError('Unexpected response format')

        # old format -> do nothing
        return result

    @staticmethod
    def _process_host_list(result):
        return [host_info['host'] for host_info, resource in result]

    def _do_request(self, url):
        exc = None
        for timeout in self.timeouts:
            try:
                r = requests.get(url, timeout=timeout, verify=False)
                if r.status_code != requests.codes.ok:
                    if r.status_code == requests.codes.not_found:
                        raise SamogonHttpError(r.status_code, 'not found', url)
                    # r.reason was not available in this version of requests, that's why
                    raise SamogonHttpError(r.status_code, '', url)
                return r.json()

            except requests.exceptions.Timeout:
                exc = 'timeout'
            except requests.exceptions.ConnectionError as e:
                exc = str(e)

        if exc == 'timeout':
            raise SamogonHttpError(0, 'timeout', url)
        if exc is not None:
            raise SamogonHttpError(0, exc, url)
