import os
import time
import ctypes
import tempfile
import errno
import threading
import re

from pkg_resources import require

require('sysv_ipc')
import sysv_ipc
import fcntl

from kernel.util.sys.syslibs import libc
from kernel.util.sys.gettime import monoTime
from kernel.util.sys.dirut import ensureDirs
from kernel.util.errors import sAssert


# ===== ctype function declaration =====
_ftok = libc(useErrNo=True).ftok
_ftok.argtypes = [ctypes.c_char_p, ctypes.c_int]
_ftok.restype = ctypes.c_int


def ftok(path, counter):
    """
    Convert a pathname and a project identifier to a System V IPC key
    @param path: path to existing accessible file
    @param counter: proj_id
    @return: int
    """
    key = _ftok(path, counter)
    if key == -1:
        errno_ = ctypes.get_errno()
        raise OSError(errno_, os.strerror(errno_))
    return key


class SemaphoreError(RuntimeError):
    """
    Base class for semaphore module exceptions
    """
    pass


class SemaphoreBusyError(SemaphoreError):
    """
    Semaphore acquision failed.
    """
    pass


class UMask(object):
    def __init__(self, umask=0):
        self.__umask = umask
        self.__oldUmask = None

    def __enter__(self):
        self.__oldUmask = os.umask(self.__umask)

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.__oldUmask is not None:
            os.umask(self.__oldUmask)

_hackGranularity = 20.0 / 1000.0
_sleep = time.sleep


class FLock(object):
    class __Busy(Exception):
        pass

    def __init__(self, fileName):
        """
        :type fileName: str
        """
        self.__fileName = fileName
        self.__fl = None

    def acquire(self, block=True):
        try:
            sAssert(self.__fl is None)

            with UMask():
                ensureDirs(os.path.dirname(self.__fileName))
                self.__fl = open(self.__fileName, 'w+')

            flags = fcntl.LOCK_EX
            if not block:
                flags |= fcntl.LOCK_NB

            fcntl.flock(self.__fl.fileno(), flags)

        except EnvironmentError as err:
            self.__cleanupFl()
            if err.errno == errno.EWOULDBLOCK:
                return False
            raise

        else:
            return True

    def release(self):
        self.__cleanupFl()

    def __cleanupFl(self):
        if self.__fl:
            try:
                self.__fl.close()
            except EnvironmentError:
                pass
            self.__fl = None

    def __enter__(self):
        return self.acquire()

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.release()


class Semaphore(object):
    """
    Interprocess semaphore that uses
    SYSV semaphore to coordinate resource usage.
    """
    def __init__(self, name, value):
        """
        :param name: Specify name for the semaphore, e.g skynet.nekto0n
        :param value: Specify initial value for the semaphore
        :return: None
        :raises: ValueError if parameters where wrong.
        """
        if not isinstance(name, basestring):
            raise ValueError("'name' must be string")

        if not isinstance(value, (int, long)):
            raise ValueError("'value' must be int")

        if not name or not value:
            error = "Bad values specified: name={0}, value={1}".format(name, value)
            raise ValueError(error)

        if value < 1:
            raise ValueError("value must be > 0, got {0}".format(value))

        self.__name = name
        self.__value = value
        self.__acquired = 0

    @property
    def name(self):
        return self.__name

    @property
    def value(self):
        return self.__value

    def acquire(self, timeout=None, delta=1):
        """
        Acquire semaphore decrementing its counter by delta
        :param delta: delta to subtract from semaphore counter
        :type delta: int
        :param timeout: timeout in seconds, None for infinity.
        :type timeout: None or int or float
        :return: True if acquired successfully, otherwise - False
        """
        if not isinstance(delta, (int, long)):
            raise ValueError("'delta' must be int")

        if not delta >= 1:
            raise ValueError("'delta' must be >=1")

        stamp = monoTime() if timeout is not None else 0

        try:
            sem = self._getSem()
            while True:
                try:
                    sem.block = timeout is None
                    sem.acquire(delta=delta)
                    self.__acquired += delta
                    return True
                except sysv_ipc.BusyError:
                    if monoTime() - stamp >= timeout:
                        return False
                    _sleep(_hackGranularity)
                except sysv_ipc.ExistentialError:
                    sem = self._getSem()

        except Exception as err:
            if not isinstance(err, SemaphoreError):
                raise SemaphoreError(str(err))
            else:
                raise

    def release(self, delta=1):
        """
        Release semaphore, incrementing its counter by delta.
        :param delta: delta to add to semaphore counter
        :type delta: int
        """

        if not isinstance(delta, (int, long)):
            raise ValueError("'delta' must be int")

        if not delta >= 1:
            raise ValueError("'delta' must be >=1")

        if delta > self.__acquired:
            raise SemaphoreError('Releasing ({0}) more that acquired({1})'.format(delta, self.__acquired))

        try:
            sem = self._getSem()
            sem.release(delta=delta)
            self.__acquired -= delta
        except Exception as err:
            if not isinstance(err, SemaphoreError):
                raise SemaphoreError(str(err))
            else:
                raise
        else:
            if self.__acquired == 0:
                self._removeSem()

    def current(self):
        """
        Get semaphore current counter.
        """
        try:
            sem = self._getSem(False)
        except sysv_ipc.ExistentialError:
            return self.value
        else:
            return sem.value

    @property
    def counter(self):
        """
        Alias for current()
        """
        return self.current()

    def _getSem(self, create=True):
        with FLock(self._fileName()):
            key = ftok(self._fileName(), self.value)

        retries = 2
        while True:
            try:
                if create:
                    sem = sysv_ipc.Semaphore(
                        key,
                        mode=0666,
                        flags=sysv_ipc.IPC_CREX,
                        initial_value=self.value
                    )
                else:
                    raise sysv_ipc.ExistentialError()

            except sysv_ipc.ExistentialError:  # already exists
                # don't set initial value if semaphore exists
                sem = sysv_ipc.Semaphore(key)

            except EnvironmentError, err:
                if err.errno == errno.ENOSPC and retries > 0:
                    self.GC()
                    retries -= 1
                    continue
                else:
                    raise

            sem.undo = True  # undo operation on process exit
            return sem

    def _removeSem(self):
        try:
            sem = self._getSem(False)
        except sysv_ipc.ExistentialError:
            return

        if sem.value != self.value:
            return

        try:
            try:
                sem.block = False
                sem.acquire(delta=self.value)
            except sysv_ipc.BusyError:
                return
            else:
                try:
                    sem.remove()
                except sysv_ipc.Error:
                    pass
                try:
                    os.remove(self._fileName())

                except EnvironmentError:
                    pass

        except Exception, err:
            if not isinstance(err, SemaphoreError):
                raise SemaphoreError(str(err))
            else:
                raise

    if os.access('/var/tmp', os.W_OK):
        workingDir = os.path.join('/var/tmp', 'semaphore')
    else:
        workingDir = os.path.join(tempfile.gettempdir(), 'semaphore')

    def _fileName(self):
        return os.path.join(
            self.workingDir,
            self.semFileNamePattern.format(hash(self.name), self.value)
        )

    def __enter__(self):
        self.acquire()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.release()

    _GCLock_ = threading.Lock()
    semFileNamePattern = '{0}_{1}.semaphore'
    semFileNameRe = re.compile('[-+]?\d+_(\d+)\.semaphore$')

    @classmethod
    def GC(cls):
        if not cls._GCLock_.acquire(False):
            #already collecting in sibling thread
            #just wait for GC in sibling thread and return
            with cls._GCLock_:
                return

        #Lock are acquired here

        try:
            for fileName in os.listdir(cls.workingDir):
                match = cls.semFileNameRe.match(fileName)

                if not match:
                    continue

                try:
                    value = int(match.group(1))
                    fullFileName = os.path.join(cls.workingDir, fileName)
                    lock = FLock(fullFileName)
                    if lock.acquire(False):
                        try:
                            key = ftok(fullFileName, value)
                            try:
                                sem = sysv_ipc.Semaphore(key)
                            except sysv_ipc.ExistentialError:
                                raise

                            if sem.value == value:
                                sem.block = False
                                sem.undo = True
                                try:
                                    sem.acquire(delta=value)
                                except sysv_ipc.BusyError:
                                    continue
                                else:
                                    try:
                                        sem.remove()
                                    except sysv_ipc.Error:
                                        continue
                                    os.remove(fullFileName)
                        except sysv_ipc.Error:
                            continue
                        finally:
                            lock.release()
                except EnvironmentError:
                    continue
        finally:
            cls._GCLock_.release()


class TimingSemaphore(Semaphore):
    """
    Semaphore with hardcoded timeout. Thus every
    call to acquire is supplied with timeout value. In case of
    timeout SemaphoreBusyError will be raised. So you can use "with" syntax.
    """
    def __init__(self, name, value, timeout):
        Semaphore.__init__(self, name, value)
        if timeout < 0:
            raise ValueError("'timeout' must be positive")
        self.timeout = timeout

    def acquire(self, delta=1):
        """
        :raises: SemaphoreBusyError, RuntimeError
        """
        res = Semaphore.acquire(self, self.timeout, delta=delta)
        if not res:
            raise SemaphoreBusyError()
        return res
