import os
import time
import yaml
import errno
import fcntl
import importlib
import py
from .errors import SkycoreError, NamespaceError, NotRunningError, PathError, ConfigNotReadyError, ServiceNotFoundError
from .moduletools import (
    _register_namespaces,
    _register_paths,
    _register_proxy_packages,
    _require_dependencies,
)
from .utils import (
    rpc_client,
    get_skycore_root, get_data_root,
    wrap_exception,
    dict2slotted, merge_recursive, Config,
    human_time
)

import six

try:
    from api.config import useGevent as _use_gevent
except ImportError:
    def _use_gevent(*args, **kwargs):
        return False


RPC_CALL_TIMEOUT = 10.


def _apply_overrides(cfg):
    overrides = py.path.local(get_data_root()).join('conf', 'overrides.d')
    if not overrides.check(exists=1, dir=1):
        return cfg

    for path in sorted(overrides.listdir()):
        if path.ext != '.yaml':
            continue

        try:
            data = yaml.load(path.open('rb'), Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
        except Exception:
            data = None

        if not data or not isinstance(data, dict):
            continue

        cfg = merge_recursive(cfg, data)

    return cfg


def query_section(path, base=None, overrides=None, as_dict=False, fix_keys=True):
    """
    Perform __local__ registry query for given section.

    :param path:         List of (sub)section names specifying path to the selected section.
                         In case of @c None passed, the whole registry content will be returned.
    :param base:         Dictionary of values to be considered as base result, its values will be
                         overridden by the ones from the fetched config.
    :param overrides:    Plain dictionary of values to be placed into result additionally
                         __relatively__ to the sub-section specified above, overriding
                         any existing data. Key can be comma-separated to represent inner dictionaries.
    :param as_dict:      Return configuration as plain dictionary instead of slotted class instance.
    :param fix_keys:     In case of slotted class instance requested, perform automatic transliteration
                         of unsafe key characters.
    :return:             Dictionary or slotted config class with the configuration data for the queried section.

    :raise PathError:    In case of no such section exists.
    """
    fmt = (
        lambda x: dict2slotted(x, fix_keys=fix_keys),
        lambda x: x,
    )[int(as_dict)]

    cfgfile = os.path.join(get_data_root(), 'conf', 'actual.yaml')
    if not os.path.isfile(cfgfile):
        raise ConfigNotReadyError("config is not ready yet")

    try:
        with open(cfgfile) as f:
            data = yaml.load(f, Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
            data = _apply_overrides(data)
    except Exception as e:
        raise PathError("config is broken: %s" % (e,))

    try:
        for p in path:
            data = data['subsections'][p]
    except (KeyError, IndexError):
        raise PathError("unknown path: %s" % (path,))

    data = merge_recursive(base, data)

    # Perform overrides merge
    if overrides:
        for k, v in six.iteritems(overrides):
            path = k.split('.')
            curr, prev = data, None
            for p in path[:-1]:
                prev, curr = curr, curr.get(p, None)
                if not isinstance(curr, dict):
                    curr = prev[p] = {}
            curr[path[-1]] = v

    return fmt(data)


class Registry(object):
    def __init__(self, gevent=None):
        if gevent is None:
            gevent = _use_gevent()
        self.__slave = rpc_client(gevent=gevent)

    @wrap_exception()
    def query_section(self, path, base=None, overrides=None, as_dict=False, fix_keys=True):
        """
        Perform __local__ registry query for given section.
        Note: this method kept for backward compatibility only. Please use query_section() function.

        :param path:         List of (sub)section names specifying path to the selected section.
                             In case of @c None passed, the whole registry content will be returned.
        :param base:         Dictionary of values to be considered as base result, its values will be
                             overridden by the ones from the fetched config.
        :param overrides:    Plain dictionary of values to be placed into result additionally
                             __relatively__ to the sub-section specified above, overriding
                             any existing data. Key can be comma-separated to represent inner dictionaries.
        :param as_dict:      Return configuration as plain dictionary instead of slotted class instance.
        :param fix_keys:     In case of slotted class instance requested, perform automatic transliteration
                             of unsafe key characters.
        :return:             Dictionary or slotted config class with the configuration data for the queried section.

        :raise PathError:    In case of no such section exists.
        """
        return query_section(path, base, overrides, as_dict, fix_keys)

    @wrap_exception()
    def pause_update(self):
        """
        Stop updating local config from genisys. For debugging purposes mostly.

        :return                     True if updating stopped, False if it has been stopped before.
        :raise AuthorizationError:  Action executed by unprivileged user.
        :raise NotRunningError:     Service is not running.
        """
        return self.__slave.call('pause').wait(RPC_CALL_TIMEOUT)

    @wrap_exception()
    def unpause_update(self):
        """
        Start updating local config from genisys. For debugging purposes mostly.

        :return                     True if updating started, False if it has been started before.
        :raise AuthorizationError:  Action executed by unprivileged user.
        :raise NotRunningError:     Service is not running.
        """
        return self.__slave.call('unpause').wait(RPC_CALL_TIMEOUT)

    @wrap_exception()
    def state(self):
        """
        Shows some internal state info.

        :return State as a dict.
        :raise NotRunningError:     Service is not running.
        """
        return self.__slave.call('state').wait(RPC_CALL_TIMEOUT)


class ServiceManager(object):
    def __init__(self, gevent=None):
        if gevent is None:
            gevent = _use_gevent()
        self.__slave = rpc_client(gevent=gevent)

    @wrap_exception()
    def get_stats(self, timeout=90):
        res = self.__slave.call(
            'get_stats',
        ).wait(timeout + RPC_CALL_TIMEOUT)
        return res

    @wrap_exception()
    def start_services(self, namespace, services=None, timeout=90.):
        """
        Start services in specified namespace.

        :param str namespace:       name of the namespace to deal with
        :param iterable services:   services to start, if empty, everything will be started
        :param float timeout:       time to wait for operation to finish
        :return                     True is service(s) have been started successfully, False otherwise
        :raise AuthorizationError:  Action executed by unprivileged user.
        :raise NamespaceError:      Specified namespace is not found.
        :raise NotRunningError:     Service is not running.
        """
        return self.__slave.call(
            'start_services',
            namespace=namespace,
            services=services,
            timeout=timeout).wait(timeout + RPC_CALL_TIMEOUT)

    @wrap_exception()
    def stop_services(self, namespace, services=None, timeout=150.):
        """
        Stop services in specified namespace.

        :param str namespace:       name of the namespace to deal with
        :param iterable services:   services to stop, if empty, everything will be stopped
        :param float timeout:       time to wait for operation to finish
        :return                     True is service(s) have been stopped successfully, False otherwise
        :raise AuthorizationError:  Action executed by unprivileged user.
        :raise NamespaceError:      Specified namespace is not found.
        :raise NotRunningError:     Service is not running.
        """
        return self.__slave.call(
            'stop_services',
            namespace=namespace,
            services=services,
            timeout=timeout).wait(timeout + RPC_CALL_TIMEOUT)

    @wrap_exception()
    def restart_services(self, namespace, services=None, timeout=300.):
        """
        Restart services in specified namespace.

        :param str namespace:       name of the namespace to deal with
        :param iterable services:   services to restart, if empty, everything will be restarted
        :param float timeout:       time to wait for operation to finish
        :return                     True is service(s) have been restarted successfully, False otherwise
        :raise AuthorizationError:  Action executed by unprivileged user.
        :raise NamespaceError:      Specified namespace is not found.
        :raise NotRunningError:     Service is not running.
        """
        return self.__slave.call(
            'restart_services',
            namespace=namespace,
            services=services,
            timeout=timeout).wait(timeout + RPC_CALL_TIMEOUT)

    @wrap_exception()
    def check_services(self, namespace, services=None, new_format=False):
        """
        Check state of services in specified namespace

        :param str namespace:        name of the namespace to deal with
        :param iterable services:    services to check, if empty, list everything
        :param bool new_format:      service state and other info in dict format (old format == list)
        :return                      dict of services and their current states
        :raise NamespaceError:       specified namespace is not found.
        :raise ServiceNotFoundError: some requested services not found in namespace
        """
        state_file = os.path.join(get_data_root(), 'skycore.state')
        if not os.path.isfile(state_file):
            raise SkycoreError("state file with services is not available")

        try:
            mtime = os.stat(state_file).st_mtime
            boot_time = time.time() - float(open('/proc/uptime').read().split()[0])
        except Exception as e:
            # cannot get boot time or mtime, assume they are ok
            mtime = 1
            boot_time = 0

        if boot_time > mtime:
            raise SkycoreError("state file is older than boot time, data is not reliable")

        try:
            with open(state_file) as f:
                try:
                    fcntl.flock(f.fileno(), fcntl.LOCK_EX | fcntl.LOCK_NB)
                except IOError as e:
                    if e.errno not in (errno.EACCES, errno.EWOULDBLOCK):
                        raise
                else:
                    raise Exception("file is not locked, skycore probably is not running")
                state = yaml.load(f, Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
        except Exception as e:
            raise SkycoreError("cannot read list of services, state-file is not readable: %s" % (e,))

        if not isinstance(state, dict):
            raise SkycoreError("skycore state is corrupted (dict expected, got %s)" % (type(state).__name__))

        state.pop('__version__', None)

        if namespace not in state:
            raise NamespaceError("namespace %r not found" % (namespace,))

        namespace = state[namespace]

        if not isinstance(namespace, dict):
            raise SkycoreError("skycore state is corrupted (dict expected, got %s)" % (type(namespace).__name__))

        try:
            srvcs = {s['service']: s for s in namespace['services']}
        except KeyError:
            raise SkycoreError("skycore state is corrupted (not all services in namespace have `service` field)")

        if services:
            not_found = filter(lambda name: name not in srvcs, services)
            if not_found:
                raise ServiceNotFoundError("Service(s): %s not found in this namespace" % (not_found,))

            srvcs = {name: srvcs[name] for name in services}

        if new_format:
            return {
                name: {
                    'state': svc['state'],
                    'version': svc['meta'].get('version', None),
                    'state_uptime': human_time(time.time() - svc.get('state_start_time', time.time() + 1)),
                } for name, svc in six.iteritems(srvcs)
            }

        return {
            name: (svc['state'], svc['meta'].get('version', None),)
            for name, svc in six.iteritems(srvcs)
        }

    @wrap_exception()
    def list_namespaces(self):
        """
        List all known namespaces

        :return                     List of namespaces
        :raise SkycoreError:        Unavailable to get list of namespaces
        """
        state_file = os.path.join(get_data_root(), 'skycore.state')
        if not os.path.isfile(state_file):
            raise SkycoreError("state file with namespaces is not available")

        try:
            with open(state_file) as f:
                state = yaml.load(f, Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
        except Exception as e:
            raise SkycoreError("cannot read list of namespaces, state-file is not readable: %s" % (e,))

        if not isinstance(state, dict):
            raise SkycoreError("skycore state is corrupted (dict expected, got %s)" % (type(state).__name__))

        internal_keys = ('__version__',)
        return [key for key in six.iterkeys(state) if key not in internal_keys]

    @wrap_exception(NamespaceError)
    def list_services(self, namespace):
        """
        List all services in the specified namespace

        :param str namespace:       name of the namespace to deal with
        :return                     List of services
        :raise NamespaceError:      Specified namespace is not found.
        :raise SkycoreError:        Unavailalble to get list of services
        """
        state_file = os.path.join(get_data_root(), 'skycore.state')
        if not os.path.isfile(state_file):
            raise SkycoreError("state file with services is not available")

        try:
            with open(state_file) as f:
                state = yaml.load(f, Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
        except Exception as e:
            raise SkycoreError("cannot read list of services, state-file is not readable: %s" % (e,))

        if not isinstance(state, dict):
            raise SkycoreError("skycore state is corrupted (dict expected, got %s)" % (type(state).__name__))

        state.pop('__version__', None)

        if namespace not in state:
            raise NamespaceError("namespace %r not found" % (namespace,))

        namespace = state[namespace]

        if not isinstance(namespace, dict):
            raise SkycoreError("skycore state is corrupted (dict expected, got %s)" % (type(namespace).__name__))

        try:
            services = [service['service'] for service in namespace['services']]
        except KeyError:
            raise SkycoreError("skycore state is corrupted (not all services in namespace have `service` field)")

        return services

    @wrap_exception()
    def install_tgz(self, namespace, path, timeout=300.):
        """
        Inistall new service to specified namespace

        :param str namespace:       name of the namespace to deal with
        :param str path:            absolute path to tgz file
        :param float timeout:       time to wait for operation to finish
        :return                     True if service has been installed, False otherwise
        :raise AuthorizationError:  Action executed by unprivileged user.
        :raise NamespaceError:      Specified namespace is not found.
        :raise NotRunningError:     Service is not running.
        """
        return self.__slave.call(
            'install_tgz',
            namespace=namespace,
            path=os.path.abspath(path)).wait(timeout + RPC_CALL_TIMEOUT)

    @wrap_exception()
    def uninstall_services(self, namespace, services, timeout=300.):
        """
        Uninistall specified service(s) from specified namespace

        :param str namespace:       name of the namespace to deal with
        :param iterable services:   services to remove
        :return                     True if service(s) have been uninstalled, False otherwise
        :raise AuthorizationError:  Action executed by unprivileged user.
        :raise NamespaceError:      Specified namespace is not found.
        :raise NotRunningError:     Service is not running.
        """
        return self.__slave.call(
            'uninstall_services',
            namespace=namespace,
            services=services).wait(timeout + RPC_CALL_TIMEOUT)

    def get_service_api(self, namespace, service, kind, timeout=None):
        """
        Provides service API.

        :param str namespace:       name of the namespace to deal with
        :param str service:         name of the service to deal with
        :param str kind:            type of the API requested
        :param float timeout:       obsolete, not used
        :return                     API
        :raise NamespaceError:      Specified namespace is not found.
        :raise KeyError:            Service API or kind not found
        :raise ValueError:          Service API is broken
        """
        apidir = os.path.join(get_data_root(), 'api', namespace)
        if not os.path.isdir(apidir):
            raise NamespaceError("namespace %r not found" % (namespace,))
        apifile = os.path.join(apidir, service + '.api')
        if not os.path.isfile(apifile):
            raise KeyError("API for %r/%r not found" % (namespace, service))

        try:
            with open(apifile) as f:
                api = yaml.load(f, Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))

            if not isinstance(api, dict):
                raise Exception("API is not a dict")
        except Exception as e:
            # FIXME (torkve) invent proper exception type
            raise ValueError("API is corrupted: %s" % (e,))

        if kind not in api:
            raise KeyError("API kind %r not found for %r/%r" % (kind, namespace, service))

        return api[kind]

    @wrap_exception(ImportError, NamespaceError)
    def get_service_python_api(self, namespace, service, kind=None, timeout=None):
        """
        Provides service python API.

        :param str namespace:       name of the namespace to deal with
        :param str service:         name of the service to deal with
        :param str kind:            type of the API requested
        :param float timeout:       obsolete, not used
        :return                     API
        :raise NamespaceError:      Specified namespace is not found.
        :raise KeyError:            Service API or kind not found
        :raise ValueError:          Service API is broken
        """
        try:
            api = self.get_service_api(namespace, service, kind or 'python')
        except Exception as e:
            raise ImportError("Service %r/%r has no python API (%s)" % (namespace, service, e))

        if not isinstance(api, dict):
            raise ImportError("Service %r/%r has malformed python API" % (namespace, service))

        paths = api.get('import_paths', [])
        requires = api.get('requires') or ()
        fake_namespaces = api.get('fake_namespaces') or {}
        proxy_packages = api.get('proxy_packages') or {}
        module = api.get('module')
        obj = api.get('object')
        is_callable = api.get('call', False)
        args = api.get('args') or ()
        kwargs = api.get('kwargs') or {}
        if not isinstance(paths, list) or not isinstance(module, str):
            raise ImportError("Service %r/%r has malformed python API" % (namespace, service))

        _register_namespaces(fake_namespaces)
        _register_paths(paths)
        _register_proxy_packages(proxy_packages)
        _require_dependencies(requires)

        try:
            mod = importlib.import_module(module)
            if obj is None:
                return mod
            elif is_callable:
                return getattr(mod, obj)(*args, **kwargs)
            else:
                return getattr(mod, obj)
        except Exception as e:
            raise ImportError("Service %r/%r has malformed python API: %r" % (namespace, service, e))

    @wrap_exception()
    def get_service_field(self, namespace, service, field, raw=False, timeout=None):
        """Get arbitrary service descriptor field"""
        fieldsdir = os.path.join(get_data_root(), 'api', namespace)
        if os.path.isdir(fieldsdir):
            fieldsfile = os.path.join(fieldsdir, service + '.fields')
            if os.path.isfile(fieldsfile):
                try:
                    with open(fieldsfile) as f:
                        fields = yaml.load(f, Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
                except Exception:
                    pass
                else:
                    if isinstance(fields, dict):
                        fields = fields['raw' if raw else 'rendered']
                        return fields[field]
        try:
            call = wrap_exception()(self.__slave.call)(
                'get_service_field',
                namespace=namespace,
                service=service,
                field=field,
                raw=raw)
            wait = wrap_exception()(call.wait)
            field = wait(timeout if timeout is not None else RPC_CALL_TIMEOUT)
        except KeyError:
            raise KeyError("There's no service %r/%r in skycore" % (namespace, service))
        except NotRunningError:
            raise KeyError("Skycore is not running, cannot get api for %r/%r" % (namespace, service))

        return field
