import os
import sys
import types
import importlib
import pkg_resources

import six


def _traverse(modname):
    items = modname.split('.')
    for parts in six.moves.xrange(1, len(items) + 1):
        yield '.'.join(items[:parts])


def _make_stub(modname):
    if modname in sys.modules:
        return sys.modules[modname]
    stub = sys.modules[modname] = types.ModuleType(modname, 'Fake module generated by skycore')
    if '.' in modname:
        parent, shortname = modname.rsplit('.')
        setattr(importlib.import_module(parent), shortname, stub)

    return stub


def _register_namespaces(fake_namespaces):
    if not fake_namespaces:
        return
    for name, paths in six.iteritems(fake_namespaces):
        if isinstance(paths, str):
            paths = [paths]

        for part_name in _traverse(name):
            mod = _make_stub(part_name)
            modpath = mod.__dict__.setdefault('__path__', [])

            pkg_resources.declare_namespace(part_name)

        modpath.extend(paths)


def _register_paths(paths):
    if not paths:
        return
    if not all(path in sys.path for path in paths):
        sys.path[:0] = paths
        for path in paths:
            pkg_resources.working_set.add_entry(path)


class _LibraryProxyLoader(object):
    def __init__(self, from_root, to_root):
        self.from_root = from_root
        self.to_root = to_root

    def load_original_module(self, fullname):
        name = self.from_root + fullname[len(self.to_root):]
        mod = importlib.import_module(name)
        return mod

    def load_module(self, fullname):
        if fullname in sys.modules:
            return sys.modules[fullname]

        name = self.to_root + fullname[len(self.from_root):]
        mod = importlib.import_module(name)
        sys.modules[fullname] = mod
        return mod


class _LibraryModProxy(types.ModuleType):
    def __init__(self, name, from_root, to_root):
        super(_LibraryModProxy, self).__init__(name)
        self.__loader__ = _LibraryProxyLoader(from_root, to_root)

    def __getattr__(self, name):
        mod = self.__loader__.load_original_module(self.__name__)
        return getattr(mod, name)

    def __dir__(self):
        mod = self.__loader__.load_original_module(self.__name__)
        return dir(mod)


def _make_proxy(from_root, to_root, modname):
    m = modname[:-9] if modname.endswith('.__init__') else modname
    if m == from_root or m.startswith(from_root + '.'):
        proxyname = to_root + m[len(from_root):]
        proxy = _LibraryModProxy(proxyname, from_root, to_root)
        if '.' in proxyname:
            parent, shortname = proxyname.rsplit('.', 1)
            setattr(_make_stub(parent), shortname, proxy)

        sys.modules[proxyname] = proxy


def _make_proxy_arcadia(loader, from_root, to_root):
    for modname in sorted(list(globals()['__loader__'].memory)):
        _make_proxy(from_root, to_root, modname)


def _make_proxy_egg(loader, from_root, to_root):
    def _make_modname(filename):
        if filename.endswith('.pyc'):
            filename = filename[:-4]
        elif filename.endswith('.py'):
            filename = filename[:-3]
        else:
            return

        modname = filename.split(os.path.sep)
        if modname[-1] == '__init__':
            modname.pop()

        return '.'.join(modname)

    for modname in sorted(set(filter(None, map(_make_modname, loader._files.keys())))):
        _make_proxy(from_root, to_root, modname)


def _register_proxy_packages(proxy_packages):
    if not proxy_packages:
        return
    standalone = getattr(sys, 'is_standalone_binary', False)
    loader = globals().get('__loader__')
    for original_module, target_module in six.iteritems(proxy_packages):
        for modname in list(_traverse(target_module))[:-1]:
            _make_stub(modname)

        if loader is None:
            _make_proxy(original_module, target_module, original_module)
            continue

        if standalone:
            _make_proxy_arcadia(loader, original_module, target_module)
            continue

        from zipimport import zipimporter
        if isinstance(loader, zipimporter):
            _make_proxy_egg(loader, original_module, target_module)


def _require_dependencies(requirements):
    for requirement in requirements:
        pkg_resources.require(requirement)
