from __future__ import absolute_import
import os
import sys
import six
import copy
import socket
import errno
from functools import wraps
from kernel.util.functional import memoized, threadsafe
import api.skycore.errors


class Config(dict):
    """
    The common parent class for any configuration slotted class' instances, which will
    be produced by @c query() function, commonly designed to distinguish them from plain dictionaries.
    """
    __slots__ = []


def merge_recursive(base, other):
    if not isinstance(other, dict):
        return other
    if not isinstance(base, dict):
        return copy.deepcopy(other)
    result = copy.deepcopy(base)
    for k, v in other.iteritems():
        if isinstance(result.get(k), dict):
            result[k] = merge_recursive(result[k], v)
        else:
            result[k] = copy.deepcopy(v)
    return result


def dict2slotted(dct, fix_keys=True):
    """
    Internal function for dictionary to slotted class instance transformation.
    :param dct:     Dictionary to be processed.
    :param fix_keys: Flags the dictionary keys should be transliterated to be safe Python identifier.
    :return:        Slotted class instance.
    """
    import re

    def str_key(k):
        return k.encode('utf-8') if six.PY2 and isinstance(k, unicode) else str(k)

    re_key_esc = re.compile(r'(^[^a-z_]|[^a-z_0-9]|^\s*$)', re.IGNORECASE)
    keys = [str_key(k) if not fix_keys else re_key_esc.sub(r'_', str_key(k)) for k in dct.keys()]

    try:
        class _Config(Config):
            __slots__ = keys

            def __init__(self):
                super(_Config, self).__init__(dct)
    except TypeError:
        raise TypeError('Unable to set %r as class slots: not an identifier detected.' % keys)

    c = _Config()
    for key, value in dct.items():
        setattr(
            c,
            str_key(key) if not fix_keys else re_key_esc.sub(r'_', str_key(key)),
            dict2slotted(value, fix_keys) if isinstance(value, dict) else value
        )
    return c


def wrap_exception(*known):
    def wrap_exception_in(func):
        @wraps(func)
        def _wrap_exception(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except socket.error as ex:
                if ex.errno in (errno.ECONNREFUSED, errno.ENOENT):
                    raise api.skycore.errors.NotRunningError(ex)
                else:
                    raise
            except Exception:
                e_class, e_val, e_tb = sys.exc_info()

                if e_class in known or e_class.__name__ in known:
                    six.reraise(e_class, e_val, e_tb)
                if e_class.__name__ == 'CallFail':
                    e_new_class = type(e_val.args[3])
                    e_val = e_val.args[3]
                elif e_class.__name__ == 'CallTimeout':
                    e_new_class = api.skycore.errors.CallTimeout
                    e_val.__class__ = e_new_class
                elif e_class is KeyError:  # special case, because KeyError has terrible __str__
                    e_new_class = api.skycore.errors.SkycoreError
                    e_val = e_new_class("KeyError: %s" % (e_val.message,))
                else:
                    e_new_class = api.skycore.errors.SkycoreError
                    e_val = e_new_class(e_val)

                six.reraise(e_new_class, e_val, e_tb)

        return _wrap_exception
    return wrap_exception_in


def _require():
    import types
    import pkg_resources

    for name, path in (
        ('ya', None),
        ('ya.skynet', None),
        ('ya.skynet.services', os.path.join(get_skycore_root(), 'lib')),
    ):
        mod = sys.modules.setdefault(name, types.ModuleType(name))
        modpath = mod.__dict__.setdefault('__path__', [])
        if path and path not in modpath:
            modpath.append(path)

        if '.' in name:
            base, this = name.rsplit('.', 1)
            setattr(sys.modules[base], this, mod)

        pkg_resources.declare_namespace(name)


@threadsafe
@memoized
def rpc_client(gevent=False):
    _require()

    if gevent:
        from ya.skynet.services.skycore.rpc.gevent_client import RPCClientGevent
        return RPCClientGevent(get_sock_path(), None)
    else:
        from ya.skynet.services.skycore.rpc.client import RPCClient
        return RPCClient(get_sock_path(), None)


def _get_path_candidates():
    yield os.path.abspath(__file__)

    npath = [p for p in os.path.abspath(__file__).split(os.sep) if p]
    if __file__.startswith('/') and npath and npath[0] == 'skynet' and os.path.islink('/skynet'):
        yield os.path.join(
            os.readlink('/skynet'),
            os.sep.join(npath[1:])
        )

    yield os.path.realpath(__file__)


@memoized
def get_skycore_root():
    if 'SKYCORE_ROOT_PATH' in os.environ and os.path.exists(os.environ['SKYCORE_ROOT_PATH']):
        return os.path.abspath(os.environ['SKYCORE_ROOT_PATH'])

    for candidate in _get_path_candidates():
        # /skynet/api/skycore/../../
        root_folder = os.path.dirname(os.path.dirname(os.path.dirname(candidate)))

        next_folder = os.path.dirname(root_folder)
        while root_folder != next_folder:
            path = os.path.join(root_folder, 'skycore')
            if os.path.exists(path) and os.path.isdir(path):
                return path

            root_folder = next_folder
            next_folder = os.path.dirname(root_folder)

    raise RuntimeError('Cannot find skycore root dir')


def get_data_root():
    if 'SKYCORE_DATA_PATH' in os.environ and os.path.exists(os.environ['SKYCORE_DATA_PATH']):
        return os.path.abspath(os.environ['SKYCORE_DATA_PATH'])

    for candidate in _get_path_candidates():
        # /Berkanavt/supervisor/base/active/api/skycore/../../../../
        root_folder = os.path.abspath(os.path.join(os.path.dirname(candidate), '../../../..'))

        next_folder = os.path.dirname(root_folder)
        while root_folder != next_folder:
            path = os.path.join(root_folder, 'skycore')
            if os.path.exists(path) and os.path.isdir(path):
                return path

            root_folder = next_folder
            next_folder = os.path.dirname(root_folder)

    raise RuntimeError('Cannot find skycore data dir')


@memoized
def get_sock_path():
    if 'SKYCORE_RPC_SOCK' in os.environ:
        return os.environ['SKYCORE_RPC_SOCK']
    return os.path.join(get_data_root(), 'rpc.sock')


def human_time(s):
    if s < 0:
        return ''

    if s < 1:
        return '%dms' % (s * 1000, )

    days, hours = divmod(s, 86400)
    hours, minutes = divmod(hours, 3600)
    minutes, seconds = divmod(minutes, 60)

    if not days and not hours and not minutes:
        return '%ds' % (seconds, )
    elif not days and not hours:
        return '%dm%.2ds' % (minutes, seconds)
    elif not days:
        return '%dh%.2dm%.2ds' % (hours, minutes, seconds)
    else:
        return '%dd%.2dh%.2dm%.2ds' % (days, hours, minutes, seconds)
