import sys

try:
    __import__('pkg_resources').require('requests')
    import requests
    requests.packages.urllib3.disable_warnings()
except BaseException:
    if getattr(sys, 'is_standalone_binary', False):
        raise


class WalleHttpError(Exception):
    def __init__(self, code, reason):
        self.code = code
        self.reason = reason
        if code:
            super(WalleHttpError, self).__init__('Walle api http error: {0} {1}'.format(code, reason))
        else:
            super(WalleHttpError, self).__init__('Walle api http error: {0}'.format(reason))


class WalleBadRequestError(Exception):
    def __init__(self, msg):
        super(WalleBadRequestError, self).__init__("Walle error: '{}'".format(msg))


class WalleDnsError(Exception):
    def __init__(self):
        super(WalleDnsError, self).__init__("Walle error: failed to resolve Wall-e service")


class Walle(object):
    DEFAULT_LIMIT = 10000
    WALLE_FQDN = 'api.wall-e.yandex-team.ru'
    API_URL_HOSTS = 'https://{}/v1/hosts?fields=name,ips,inv&{}={}&cursor={}&limit={}'

    KNOWN_DC_MAP = {
        'SAS': "RU|SAS",
        'MAN': "FI|MANTSALA",
        'VLA': "RU|VLADIMIR",
        'MYT': "RU|MYT",
        'IVA': "RU|IVA"
    }

    def __init__(self):
        self.timeouts = [10, 20, 40, 60]

    def get_hosts(self, project_name):
        return self._get_hosts('project', project_name)

    def get_hosts_by_tag(self, tag):
        return self._get_hosts('tags', tag)

    def get_hosts_by_switch(self, switch, host_num):
        return self._get_hosts('switch', switch, host_num)

    def get_hosts_by_dc(self, dc):
        return self._get_hosts('physical_location', self.KNOWN_DC_MAP.get(dc.upper(), 'UNKNOWN'))

    @staticmethod
    def _process_request_value(value):
        chunks = value.split(':')
        if len(chunks) > 1:
            return chunks[0], chunks[1]

        return chunks[0], 'name'

    @staticmethod
    def _get_walle_ip():
        try:
            from api.skycore import Registry
            return Registry().query_section(['skynet', 'tools', 'sky'])['config']['config']['WalleApiIP']
        except (KeyError, ImportError):
            return None

    def _get_hosts(self, request_type, request_value, host_num=None):
        all_hosts = []
        cursor = 0

        value, opt = self._process_request_value(request_value)
        hostname = self.WALLE_FQDN
        verify = True

        while True:
            url = self.API_URL_HOSTS.format(hostname, request_type, value, cursor, self.DEFAULT_LIMIT)
            try:
                response = self._do_request(url, verify)
            except WalleDnsError:
                if verify:
                    # first time failed DNS resolving
                    # try IP address instead
                    hostname = '[{}]'.format(self._get_walle_ip())
                    verify = False
                    continue
                raise WalleDnsError()

            result = response['result']

            if not result:
                break

            for host in result:
                if host_num is not None:
                    if host_num <= 0:
                        break
                    host_num -= 1

                if opt in host:
                    v = host[opt]
                    if isinstance(v, list) and v:
                        # use only first value in list
                        v = v[0]
                    if v:
                        all_hosts.append(v)

            if len(result) < self.DEFAULT_LIMIT:
                break

            cursor = result[-1]['inv'] + 1

        return all_hosts

    def _do_request(self, url, verify):
        exc = None
        for timeout in self.timeouts:
            try:
                r = requests.get(url, timeout=timeout, verify=verify)
                exc = None
                break
            except requests.exceptions.Timeout:
                exc = 'timeout'
            except requests.exceptions.ConnectionError as e:
                exc = str(e)

        if exc is not None:
            if 'Name or service not known' in str(exc):
                raise WalleDnsError()
            else:
                raise WalleHttpError(0, exc)

        if r.status_code != requests.codes.ok:
            if r.status_code == requests.codes.bad_request:
                try:
                    msg = r.json()['message']
                except BaseException:
                    msg = 'Unknown error'
                raise WalleBadRequestError(msg)
            # r.reason was not available in this version of requests, that's why
            raise WalleHttpError(r.status_code, "")

        return r.json()
