import requests
import json

from api.skycore import Registry

YP_URL_FORMAT = '{}/{}/{}'
YP_OBJECT_SERVICE = 'ObjectService'
YP_GET_OBJECT_METHOD = 'GetObject'
YP_SELECT_OBJECTS_METHOD = 'SelectObjects'
YP_RESPONSE_MSG = 'X-YT-Response-Message'


class YPResolverError(Exception):
    def __init__(self, error, message):
        super(YPResolverError, self).__init__("{}: {}".format(error, message))


class YPServiceNotFoundError(YPResolverError):
    pass


# Path:
#   - project
#   - project.stage
#   - project.stage.du
#   - project.stage.du.box
# Y.Deploy boxes
#   b@path[?cluster]
# Y.Deploy pods
#   p@path[:transient|persistent][?cluster]
class YPPath(object):
    CLUSTER = 0
    FQDN_TYPE = 1
    PROJECT = 2
    STAGE = 3
    DU = 4
    BOX = 5
    DEFAULT_CLUSTER = 'xdc'
    DEFAULT_FQDN_TYPE = 'transient'

    def __init__(self, path):
        if '?' in path:
            path, self.cluster = path.split('?')
        else:
            self.cluster = self.DEFAULT_CLUSTER

        if ':' in path:
            path, self.fqdn_type = path.split(':')
        else:
            self.fqdn_type = self.DEFAULT_FQDN_TYPE

        self.parts = path.split('.')

    def get(self, part_id):
        if part_id == YPPath.CLUSTER:
            return self.cluster

        if part_id == YPPath.FQDN_TYPE:
            return self.fqdn_type

        # first two ids are not in self.parts
        if part_id - 2 >= len(self.parts):
            return

        return self.parts[part_id - 2]

    def has(self, part_id):
        # first two ids exists always
        return part_id - 2 < len(self.parts)


class YPClient(object):
    def __init__(self, token, master_urls):
        self.timeouts = [5, 5, 10]
        self.token = token
        self.master_urls = master_urls

    def select_objects(self, cluster, object_type, yp_filter, paths):
        # print 'select_objects:', cluster, object_type, yp_filter, paths
        if cluster not in self.master_urls:
            raise YPResolverError('Invalid cluster',
                                  'Unknown cluster "{}", use one of {}'.format(cluster, self.master_urls.keys()))

        url = YP_URL_FORMAT.format(self.master_urls[cluster], YP_OBJECT_SERVICE, YP_SELECT_OBJECTS_METHOD)
        request = {
            "object_type": object_type,
            "filter": {
                "query": yp_filter
            },
            "selector": {
                "paths": paths
            },
            "format": "PF_YSON"
        }
        res = self._do_request(url, request)
        if 'results' not in res:
            raise YPServiceNotFoundError('Not found', '{} with filter "{}"'.format(object_type, yp_filter))

        return [value['value_payloads'][0]['yson'] for value in res['results']]

    def get_object(self, cluster, object_type, object_id, paths):
        # print 'get_object:', cluster, object_type, object_id, paths
        if cluster not in self.master_urls:
            raise YPResolverError('Invalid cluster',
                                  'Unknown cluster "{}", use one of {}'.format(cluster, self.master_urls.keys()))

        url = YP_URL_FORMAT.format(self.master_urls[cluster], YP_OBJECT_SERVICE, YP_GET_OBJECT_METHOD)
        request = {
            "object_type": object_type,
            "object_id": object_id,
            "selector": {"paths": paths},
            "format": "PF_YSON"
        }
        res = self._do_request(url, request)
        if 'result' not in res:
            raise YPServiceNotFoundError('Not found', '{} "{}"'.format(object_type, object_id))

        return res['result']['value_payloads'][0]['yson']

    def _do_request(self, url, request):
        exc = None
        for timeout in self.timeouts:
            try:
                r = requests.post(
                    url,
                    json.dumps(request),
                    headers={
                        "Accept": "application/json",
                        "Content-Type": "application/json",
                        'Authorization': 'OAuth ' + self.token,
                    },
                    timeout=timeout
                )
                exc = None
                break
            except requests.exceptions.Timeout:
                exc = 'timeout'
            except requests.exceptions.ConnectionError as e:
                exc = str(e)

        if exc is not None:
            raise YPResolverError('HTTP error', '{} for {}'.format(exc, url))

        if r.status_code != requests.codes.ok:
            try:
                msg = r.headers[YP_RESPONSE_MSG]
            except BaseException:
                msg = 'Unknown error'

            # r.reason was not available in this version of requests, that's why
            raise YPResolverError('HTTP error', '{} {} for {}'.format(r.status_code, msg, url))

        return r.json()


class SDClient(object):
    SD_RESOLVE_PODS_URL = 'http://sd.yandex.net:8080/resolve_pods/json'

    class EResolveStatus(object):
        NOT_EXISTS = 0
        NOT_CHANGED = 1
        OK = 2
        EMPTY = 3

    def __init__(self):
        self.timeouts = [5, 5, 10]

    def resolve_pods(self, cluster_name, pod_set_id, pod_label_selectors=None):
        request = {
            "cluster_name": cluster_name,
            "pod_set_id": pod_set_id,
            "client_name": "skynet.hostresolver",
        }
        if pod_label_selectors:
            request['pod_label_selectors'] = pod_label_selectors

        res = self._do_request(self.SD_RESOLVE_PODS_URL, request)
        status = res.get('resolve_status', 0)
        if status in (self.EResolveStatus.NOT_EXISTS, self.EResolveStatus.EMPTY):
            raise YPServiceNotFoundError('Not found', 'Podset: {}, cluster: {}'.format(pod_set_id, cluster_name))

        if status == self.EResolveStatus.NOT_CHANGED:
            raise YPResolverError('API Error', 'Unexpected status NOT_CHANGED received from SD server')

        return res.get('pod_set', {}).get('pods', [])

    def _do_request(self, url, request):
        exc = None
        for timeout in self.timeouts:
            try:
                r = requests.post(
                    url,
                    json.dumps(request),
                    headers={
                        "Accept": "application/json",
                        "Content-Type": "application/json",
                    },
                    timeout=timeout,
                )
                exc = None
                break
            except requests.exceptions.Timeout:
                exc = 'timeout'
            except requests.exceptions.ConnectionError as e:
                exc = str(e)

        if exc is not None:
            raise YPResolverError('HTTP error', '{} for {}'.format(exc, url))

        if r.status_code != requests.codes.ok:
            # does SD really ever return this header?
            try:
                msg = r.headers[YP_RESPONSE_MSG]
            except BaseException:
                msg = 'Unknown error: ' + r.text

            # r.reason was not available in this version of requests, that's why
            raise YPResolverError('HTTP error', '{} {} for {}'.format(r.status_code, msg, url))

        return r.json()


class YPResolver(object):
    WILDCARD = '*'
    MODE_MAP = {
        'transient': '/status/dns/transient_fqdn',
        'persistent': '/status/dns/persistent_fqdn',
        'node': '/spec/node_id'}

    def __init__(self, use_sd_resolver=None):
        super(YPResolver, self).__init__()
        self.registry = Registry()
        self.use_sd_resolver = use_sd_resolver

    def get_boxes(self, path):
        yp_path = YPPath(path)
        yp_selector = '/status/agent/pod_agent_payload/status/boxes'
        sd_selector = self._convert_fqdn_type_to_sd_getter('persistent')
        box_type = yp_path.get(YPPath.BOX) if yp_path.has(YPPath.BOX) else 'default'
        return self._get_pod_infos(yp_path, yp_selector, sd_selector, box_type=box_type)

    def get_pods_ex(self, path):
        yp_path = YPPath(path)
        if yp_path.get(YPPath.BOX):
            raise YPResolverError('Usage error', "Pod resolve clause {!r} cannot contain box: {!r} found".format(path, yp_path.get(YPPath.BOX)))

        fqdn_type = yp_path.get(YPPath.FQDN_TYPE)
        yp_selector = self._convert_fqdn_type_to_path(fqdn_type)
        sd_selector = self._convert_fqdn_type_to_sd_getter(fqdn_type)

        return self._get_pod_infos(yp_path, yp_selector, sd_selector)

    def _get_pods_sd(self, pod_set_id, dcs, value_getter):
        sd_client = SDClient()

        pods = set()
        podset_found = False

        for dc in dcs:
            try:
                res = sd_client.resolve_pods(dc, pod_set_id)
            except YPResolverError:
                continue

            podset_found = True
            for pod in res:
                pods.add(value_getter(pod))

        if not podset_found:
            raise YPServiceNotFoundError('Not found', 'Podset: {}'.format(service))

        pods.discard(None)

        return pods

    def get_pods(self, service_id):
        pods = set()
        podset_found = False
        # return: set of pods
        service, selector, sd_selector = self._parse_value(service_id)

        token, master_urls, master_urls_per_dc, use_sd_resolver = self._read_config()
        yp_client = YPClient(token, master_urls_per_dc)

        if use_sd_resolver:
            dcs = [k for k, v in master_urls_per_dc.items() if v in master_urls]
            return self._get_pods_sd(service, dcs, sd_selector)

        for dc, url in master_urls_per_dc.items():
            # use only old masters url
            if url not in master_urls:
                continue

            # verify podset
            try:
                yp_client.get_object(dc, 'pod_set', service, ['/meta'])
            except YPResolverError:
                continue

            podset_found = True
            # get pods
            res = yp_client.select_objects(dc, 'pod', '[/meta/pod_set_id]="{}"'.format(service), [selector])
            if res:
                pods.update(res)

        if not podset_found:
            raise YPServiceNotFoundError('Not found', 'Podset: {}'.format(service))

        # Remove None from pod list (it can happen if some DNS fields are missing in YP)
        if None in pods:
            pods.remove(None)

        return pods

    def _get_pod_infos(self, yp_path, yp_selector, sd_selector, box_type=None):
        token, _, master_urls_per_dc, use_sd_resolver = self._read_config()
        yp_client = YPClient(token, master_urls_per_dc)
        sd_client = SDClient()

        if not yp_path.has(YPPath.PROJECT):
            raise YPResolverError('Invalid path', 'please specify project')

        cluster = yp_path.get(YPPath.CLUSTER)

        yp_filter = ''
        if yp_path.get(YPPath.PROJECT) != self.WILDCARD:
            yp_filter = '[/meta/project_id]="{}"'.format(yp_path.get(YPPath.PROJECT))

        if yp_path.has(YPPath.STAGE) and yp_path.get(YPPath.STAGE) != self.WILDCARD:
            yp_filter += '{}[/meta/id]="{}"'.format(' and ' if yp_filter else '', yp_path.get(YPPath.STAGE))

        if not yp_filter:
            raise YPResolverError('Invalid path', 'please specify project or stage')

        stage_infos = yp_client.select_objects(cluster,
                                               'stage',
                                               yp_filter,
                                               ['/status/deploy_units'])

        results = set()
        deploy_unit = yp_path.get(YPPath.DU)
        for deploy_units in stage_infos:
            for k, v in deploy_units.items():
                if deploy_unit and deploy_unit != self.WILDCARD and k != deploy_unit:
                    continue

                boxes = None
                if use_sd_resolver and box_type is not None:
                    ct = v['current_target']
                    pts = (
                        ct['replica_set']['replica_set_template']['pod_template_spec']
                        if 'replica_set' in ct
                        else ct['multi_cluster_replica_set']['replica_set']['pod_template_spec']
                    )
                    boxes_spec = pts['spec']['pod_agent_payload']['spec']['boxes']
                    boxes = [
                        box['id'].lower()
                        for box in boxes_spec
                        if box_type == self.WILDCARD or box['id'] == box_type or box.get('specific_type', 'default') == box_type
                    ]

                rs = v['replica_set'] if 'replica_set' in v else v['multi_cluster_replica_set']
                cluster_statuses = rs.get('cluster_statuses', {})

                for dc, value in cluster_statuses.items():
                    if use_sd_resolver:
                        # This branch in fact can return not pod infos, but boxes.
                        # This is because yp_path can match multiple stages/deploy_units with different
                        # box configurations, and this info would be lost after this method, so we would not
                        # be able to prepend box types afterwards.
                        tmp_res = sd_client.resolve_pods(dc, value['endpoint_set_id'])
                        for res in tmp_res:
                            res = sd_selector(res)
                            if res and boxes is not None:
                                results.update('{}.{}'.format(box, res) for box in boxes)
                            elif res:
                                results.add(res)
                    else:
                        tmp_res = yp_client.select_objects(
                            dc,
                            'pod',
                            '[/meta/pod_set_id]="{}"'.format(value['endpoint_set_id']),
                            [yp_selector],
                        )
                        if box_type is None:
                            results.update(res for res in tmp_res if res)
                        else:
                            for boxes in tmp_res:
                                if not boxes:
                                    continue
                                for box in boxes:
                                    # TODO: use fqdn instead of ip6_address as soon as it's available
                                    if box_type == self.WILDCARD or box['id'] == box_type or box['specific_type'] == box_type:
                                        if box['ip6_address']:
                                            results.add('[{}]'.format(box['ip6_address']))

        return results

    def _read_config(self):
        try:
            token = self.registry.query_section(['skynet', 'tools', 'sky'])['config']['config']['YpApiAuth']
        except KeyError:
            raise YPResolverError('Invalid config', 'YP API OAUTH token is not available')

        try:
            master_urls = self.registry.query_section(['skynet', 'tools', 'sky'])['config']['config']['YpMasters']
        except KeyError:
            raise YPResolverError('Invalid config', 'YP MASTER URLs are not available')

        try:
            master_urls_per_dc = self.registry.query_section(['skynet', 'tools', 'sky'])['config']['config'][
                'YpMastersPerDc']
        except KeyError:
            raise YPResolverError('Invalid config', 'YP MASTER URLs PER DC are not available')

        use_sd_resolver = self.use_sd_resolver
        if use_sd_resolver is None:
            try:
                use_sd_resolver = self.registry.query_section(['skynet', 'tools', 'sky'])['config']['config']['UseSDResolver']
            except KeyError:
                use_sd_resolver = False

        return token, master_urls, master_urls_per_dc, use_sd_resolver

    def _parse_value(self, value):
        chunks = value.split(':')
        # simple conversion Nanny service to podset
        value = chunks[0].replace('_', '-')
        mode = chunks[1] if len(chunks) > 1 else 'transient'
        return value, self._convert_fqdn_type_to_path(mode), self._convert_fqdn_type_to_sd_getter(mode)

    def _convert_fqdn_type_to_path(self, fqdn_type):
        path = self.MODE_MAP.get(fqdn_type, '')
        if not path:
            raise YPResolverError('Invalid option',
                                  'Unknown mode "{}", use one of {}'.format(fqdn_type, self.MODE_MAP.keys()))

        return path

    def _convert_fqdn_type_to_sd_getter(self, fqdn_type):
        if fqdn_type == 'transient':
            return lambda pod: pod.get('dns', {}).get('transient_fqdn')
        elif fqdn_type == 'persistent':
            return lambda pod: pod.get('dns', {}).get('persistent_fqdn')
        elif fqdn_type == 'node':
            return lambda pod: pod.get('node_id')

        raise YPResolverError('Invalid option',
                              'Unknown mode "{}", use one of {}'.format(fqdn_type, self.MODE_MAP.keys()))
