import os
import time
__import__('pkg_resources').require('Pyro4==4.8')
import Pyro4
import logging

from kernel.pyro.errors import WireError
from kernel.util.errors import raiseEx
from kernel.pyro.misc import getSocketPath


__all__ = ['Client']


log = logging.getLogger(__name__)


class PyroMethodProxy(object):
    def __init__(self, pyroUri, pyroMethod, timeouts):
        self.pyroUri = pyroUri
        self.pyroMethod = pyroMethod
        self.timeouts = timeouts

    def __call__(self, *args, **kwargs):
        try:
            for attempt in self.timeouts:
                try:
                    return self.pyroMethod(*args, **kwargs)
                except Pyro4.errors.CommunicationError as err:
                    if len(self.timeouts) > 1:
                        log.debug('got %r from %s! Retry in %.1f seconds' % (err, self.pyroUri, attempt))
                    time.sleep(attempt)
            raise
        except Pyro4.errors.CommunicationError as err:
            log.error('got %r from %s after %r attempts' % (err, self.pyroUri, self.timeouts))
            params = self.pyroUri, err.__class__.__name__, err, len(self.timeouts)
            message = 'connection to %s failed (%s: %s), tried %s times' % params
            raiseEx(WireError(message), err, 'PyroMethodProxy')


class PyroProxy(Pyro4.Proxy):
    def __init__(self, *args, **kwargs):
        timeouts = kwargs.pop('retries', [0.1]*10)
        self.timeouts = timeouts
        super(PyroProxy, self).__init__(*args, **kwargs)

    def __getattr__(self, name):
        return PyroMethodProxy(self._pyroUri, Pyro4.Proxy.__getattr__(self, name), self.timeouts)


def makeuri(name):
    """
    create pyro uri from given service name
    """
    return "PYRO:{0}@./u:{1}".format(name, getSocketPath(name))


class Func(object):
    """
    Function proxy.
    """
    def __init__(self, name, proxy):
        self.proxy = proxy
        self.name = name

    def __call__(self, *args, **kwargs):
        remotemethod = getattr(self.proxy, self.name)
        return remotemethod(*args, **kwargs)


class Client(object):
    def __init__(self, name):
        self._name = name
        self.uri = makeuri(name)
        self.proxy = self._getProxy(self.uri)

    def _getProxy(self, uri):
        return PyroProxy(uri)

    def maySendRequest(self):
        return os.path.exists(getSocketPath(self._name))

    def __getattr__(self, name):
        if name.startswith('__'):
            raise AttributeError(name)
        return Func(name, self.proxy)


class AsyncClient(Client):
    """
    Like kernel.zeromq.async - every call runs in separate thread.
    To make it one way call - use _pyroOneway.add().
    """
    def _getProxy(self, uri):
        return Pyro4.async(PyroProxy(uri))  # noqa: W606


class NoRetryClient(Client):
    """ Client with no retries on any network failures. """
    def _getProxy(self, uri):
        return PyroProxy(uri, retries=(0,))  # make no retries


CtlClient = NoRetryClient
