"""
Copies of kernel.zeromq.server.Lock to provide exactly the same interface
and kernel.zeromq.server.loop function.
"""

import os
import socket
import threading
import fcntl
import errno

__import__('pkg_resources').require('Pyro4==4.8')
import Pyro4
import logging

from kernel.pyro.misc import getSocketPath
from kernel.util.misc import safeOpen
from kernel.util.errors import formatException


__all__ = ['Lock']


log = logging.getLogger(__name__)


class Lock(object):
    def __init__(self, path, addr):
        self.path = path
        self.addr = addr
        self.fd = None
        self.good = False

    def __enter__(self):
        prefix = 'ipc://'

        if self.path.startswith(prefix):
            self.good = True
            self.path = self.path[len(prefix):] + '.lock'

        if self.good:
            self.fd = safeOpen(self.path, 'wb')

            try:
                fcntl.flock(self.fd, fcntl.LOCK_EX | fcntl.LOCK_NB)
            except IOError:
                raise Exception('address %s already bound' % self.addr)

        return self

    def __exit__(self, *args):
        if self.fd:
            os.unlink(self.path)

    def check(self):
        self.__enter__()
        return self.good


class TCPLock(object):
    def __init__(self, port, iface='localhost'):
        if not 0 <= port < 65535:
            raise ValueError("wrong port number {0}".format(port))
        self.port = port
        self.iface = iface
        self.sock = None

    def _close(self):
        if self.sock is not None:
            self.sock.close()
            self.sock = None

    def __enter__(self):
        self.sock = socket.socket()
        try:
            self.sock.bind(('localhost', self.port))
        except socket.error as err:
            self._close()
            if err.errno == errno.EADDRINUSE:
                raise Exception('address %s:%s already bound' % (self.iface, self.port))
            raise

    def __exit__(self, *args, **kw):
        self._close()


def loopInThread(name, handler, startEvent=None, serverType='thread'):
    """
    Pyro server mainloop.
    1. It's threaded. Every connection is handled in separate thread.
    2. :arg str name: must be a string identifier of service, e.g "skynet.whatever"
    3. :arg object handler: must be an object with :attr threading.Event stopEvent:
       attribute. Main thread will wait for this event and stop service.
    :ret int exitcode:
      * 0 if loop exited without errors
      * 1 if error happened
    """
    # ==== set up Pyro ====
    Pyro4.config.SERVERTYPE = serverType

    log.info('initializing...')

    path = getSocketPath(name, makeDirs=True)
    exitcode = 1
    try:
        with Lock('ipc://' + path, path):
            if os.path.exists(path):
                os.remove(path)
            try:
                d = Pyro4.Daemon(unixsocket=path)
            except socket.error:
                raise Exception('failed to create daemon: {0}'.format(formatException()))

            d.register(handler, name)

            loopThread = threading.Thread(target=d.requestLoop)
            loopThread.daemon = True
            loopThread.start()

            try:
                if startEvent is not None:
                    startEvent.set()

                handler.stopEvent.wait()
                log.info('exiting on stop request')
                exitcode = 0
            finally:
                d.shutdown()
    except Exception:
        log.critical('exiting on error: {0}'.format(formatException()))
    return exitcode
