import unittest


class CIStr(str):
    """
    Case insensitive strings.
    Performs like str except all comparisons are case insensitive.
    """
    def __new__(cls, value):
        self = str.__new__(cls, value)
        self.__lowerCaseValue = value.lower()
        return self

    def __repr__(self):
        return "CIStr(%s)" % str.__repr__(self)

    def __eq__(self, other):
        return self.__lowerCaseValue == other.lower()

    def __lt__(self, other):
        return self.__lowerCaseValue < other.lower()

    def __le__(self, other):
        return self.__lowerCaseValue <= other.lower()

    def __gt__(self, other):
        return self.__lowerCaseValue > other.lower()

    def __ne__(self, other):
        return self.__lowerCaseValue != other.lower()

    def __ge__(self, other):
        return self.__lowerCaseValue >= other.lower()

    def __cmp__(self, other):
        return cmp(self.__lowerCaseValue, other.lower())  # noqa

    def __hash__(self):
        return hash(self.__lowerCaseValue)

    def __contains__(self, other):
        return other.lower() in self.__lowerCaseValue

    def count(self, other, *args):
        return str.count(self.__lowerCaseValue, other.lower(), *args)

    def endswith(self, other, *args):
        return str.endswith(self.__lowerCaseValue, other.lower(), *args)

    def find(self, other, *args):
        return str.find(self.__lowerCaseValue, other.lower(), *args)

    def index(self, other, *args):
        return str.index(self.__lowerCaseValue, other.lower(), *args)

    def lower(self):   # Courtesy Duncan Booth
        return self.__lowerCaseValue

    def rfind(self, other, *args):
        return str.rfind(self.__lowerCaseValue, other.lower(), *args)

    def rindex(self, other, *args):
        return str.rindex(self.__lowerCaseValue, other.lower(), *args)

    def startswith(self, other, *args):
        return str.startswith(self.__lowerCaseValue, other.lower(), *args)


class TestCIStr(unittest.TestCase):
    def testSort(self):
        l = list((CIStr(s) for s in ["dlfksweAdkweDaR", "sdfnsejraAADFFFrewkrle", "AbcddSd", "erowepASwqeFDfsdf"]))
        l.sort()
        self.assertEqual(l, ["abcddsd", "dlfksweadkwedar", "erowepaswqefdfsdf", "sdfnsejraaadfffrewkrle"])
