from __future__ import absolute_import

import os
import sys
import six

import yaml
import yaml.loader
import yaml.constructor


from . import logging
from .functional import singleton, threadsafe
from .sys.user import getUserHome
from .objproxy import ObjProxy
from .net import uriloader

__default__ = object()


class Config(object):
    def __init__(self,
                 name,
                 defaults=None,
                 searchPackages=None,
                 searchDeployments=__default__,
                 overrides=None,
                 loadOrder=('defaults', 'package', 'overrides', 'disk'),
                 ):
        """
        Load YAML configuration.
        Configuration is loaded from the multiple locations and merged together
        to one dict.
        :param string name: base name of the configuration
        :param dict defaults: default configuration values
        :param list searchPackages: package names to search in. If any packages are specified, configuration will be taken from their resources
        :param list searchDeployments: list of deployments to look for. By default used global set of deployments
        :param dict overrides: configuration values to be used as overrides
        :param list loadOrder: order in which configurations are merged
        """
        self.name = name
        self.data = None

        self.overrides = overrides
        self.loadOrder = loadOrder

        self.setSearchPackages(searchPackages)
        self.setSearchDeployments(searchDeployments)
        self.setDefaults(defaults)

        self.log = logging.getLogger('ya.skynet.util.config')

    def setSearchPackages(self, searchPackages=None):
        if searchPackages is None:
            searchPackages = []
        elif isinstance(searchPackages, six.string_types):
            searchPackages = [searchPackages]
        self.searchPackages = searchPackages

    def setSearchDeployments(self, searchDeployments=__default__):
        if searchDeployments is __default__:
            self.deployments = list(deployments())
        elif isinstance(searchDeployments, six.string_types):
            self.deployments = [searchDeployments]
        elif not searchDeployments:
            self.deployments = []
        else:
            self.deployments = list(searchDeployments)

    def setDefaults(self, defaults=None):
        if defaults is None:
            defaults = {}
        elif isinstance(defaults, six.string_types):
            defaults = yaml.load(defaults, ConfigLoader)
        elif not isinstance(defaults, dict):
            raise AssertionError('Wrong defaults type')

        self.defaults = defaults

    def fileNames(self):
        yield '{0}-defaults.yaml'.format(self.name)
        for deployment in self.deployments:
            yield '{0}-{1}.yaml'.format(self.name, deployment)
        yield '{0}.yaml'.format(self.name)

    def filePaths(self):
        try:
            from api.srvmngr import getRoot as getInstallRoot
            installRoot = getInstallRoot()
        except Exception:
            pass
        else:
            yield os.path.join(installRoot, 'etc')

        yield os.path.join(getUserHome(), '.skynet', 'etc')

    def _loadUrl(self, url):
        try:
            stream = uriloader.load(url, as_stream=True)
            data = yaml.load(stream, ConfigLoader)
            if isinstance(data, dict):
                self.data.merge(data)
            elif data is None:
                pass
            else:
                raise AssertionError('Invalid data')
        except (yaml.YAMLError, AssertionError) as err:
            self.log.warn('Can`t load config {0} `{1}`: {2}'.format(self.name, url, err))
        except (EnvironmentError, ImportError, RuntimeError):
            pass

    def loadPackageConfigs(self):
        for fileName in self.fileNames():
            for searchPackage in self.searchPackages:
                fileFullPath = 'pkg_resources:{}@{}'.format(searchPackage, os.path.join('etc/', fileName))
                self._loadUrl(fileFullPath)

    def loadDiskConfigs(self):
        for fileName in self.fileNames():
            for filePath in self.filePaths():
                fileFullPath = 'file:{}'.format(os.path.join(filePath, fileName))
                self._loadUrl(fileFullPath)

    def loadOverrides(self):
        if self.overrides is not None:
            self.data.merge(self.overrides)

    @threadsafe
    def load(self, doReload=False):
        if doReload:
            self.data = None

        if self.data is not None:
            return

        self.data = ConfigDict()

        for option in self.loadOrder:
            if option == 'defaults':
                self.data.merge(self.defaults)
            elif option == 'package':
                # First load all packaged files
                self.loadPackageConfigs()
            elif option == 'disk':
                # Then files from disk
                self.loadDiskConfigs()
            elif option == 'overrides':
                self.loadOverrides()
            else:
                self.log.warn("Unknown option in config load order {}: {}".format(self.name, option))

        self.data.secondPass()

    def __getattr__(self, item):
        self.load()
        return getattr(self.data, item)


class ConfigDict(dict):
    _NoValue = object()

    _secondPassKey = "__second_pass_values__"

    def __getattr__(self, item):
        result = self.get(item, self._NoValue)

        if result is self._NoValue:
            raise AttributeError('Config doesn`t have attr {0}'.format(item))

        return result

    def merge(self, E=None, **F):
        if isinstance(E, dict):
            for k, v in six.iteritems(E):
                if isinstance(v, dict):
                    prevV = self.get(k)
                    if not isinstance(prevV, dict):
                        prevV = self.__class__()
                        self[k] = prevV
                    prevV.merge(v)
                elif isinstance(v, list) and (k == self._secondPassKey):
                    self.setdefault(self._secondPassKey, []).extend(v)
                else:
                    self[k] = v
            if F:
                dict.update(self, None, **F)
        else:
            dict.update(self, E, **F)

    def secondPass(self):
        for secondPassValue in self.get(self._secondPassKey, []):
            secondPassValue.secondPass(self)


class SecondPassValue(object):
    def secondPass(self, document):
        pass


class EvaluateValue(SecondPassValue):
    def __init__(self, exp):
        self.exp = exp
        self.proxy = None

    def secondPass(self, document):
        if self.proxy is not None:
            self.proxy.__dict__['obj'] = eval(self.exp, dict(sys.modules.copy()), document)
            self.proxy = None


class ConfigConstructor(yaml.constructor.Constructor):
    def construct_yaml_map(self, node):
        data = ConfigDict()
        yield data
        value = self.construct_mapping(node)
        data.update(value)

    def construct_ys_ev(self, node):
        result = ObjProxy(EvaluateValue(node.value))
        result.proxy = result
        yield result
        self.add_second_pass_value(result)

    def add_second_pass_value(self, value):
        self.secondPassValues.append(value)

    def __init__(self):
        yaml.constructor.Constructor.__init__(self)
        self.secondPassValues = []

    def construct_document(self, node):
        result = super(ConfigConstructor, self).construct_document(node)
        assert isinstance(result, ConfigDict), 'Wrong document type'
        result[result._secondPassKey] = self.secondPassValues
        self.secondPassValues = []
        return result

ConfigConstructor.add_constructor(
    u'tag:skynet.yandex.ru,2012:ys/ev',
    ConfigConstructor.construct_ys_ev
)

ConfigConstructor.add_constructor(
    u'tag:yaml.org,2002:python/dict',
    ConfigConstructor.construct_yaml_map)

ConfigConstructor.add_constructor(
    u'tag:yaml.org,2002:map',
    ConfigConstructor.construct_yaml_map)


class ConfigLoader(ConfigConstructor, yaml.loader.Loader):
    DEFAULT_TAGS = yaml.loader.Loader.DEFAULT_TAGS
    DEFAULT_TAGS.update({
        u'!ya!':  u'tag:skynet.yandex.ru,2012:ys/',
    })

    def __init__(self, stream):
        yaml.loader.Loader.__init__(self, stream)
        ConfigConstructor.__init__(self)


@singleton
def deployments():
    return list()


# TODO: Move this out from here
@singleton
def extendDeployments(distribution):
    try:
        data = distribution.get_metadata('deployments.yaml')
    except EnvironmentError:
        return

    if not data:
        return

    try:
        newDeployments = yaml.load(data)
    except yaml.YAMLError:
        return

    gDeployments = deployments()

    for deployment in newDeployments:
        if deployment not in gDeployments:
            gDeployments.append(deployment)
