from functools import wraps, partial
import weakref
import time
import six


def lazy(func, *resultclasses):
    """
    Turns any callable into a lazy evaluated callable. You need to give result
    classes or types -- at least one is needed so that the automatic forcing of
    the lazy evaluation code is triggered. Results are not memoized; the
    function is evaluated on every access.
    """
    class __proxy__(object):
        """
        Encapsulate a function call and act as a proxy for methods that are
        called on the result of that function. The function is not evaluated
        until one of the methods on the result is called.
        """
        __dispatch = None

        def __init__(self, args, kw):
            self.__func = func
            self.__args = args
            self.__kw = kw
            if self.__dispatch is None:
                self.__prepare_class__()

        def __prepare_class__(cls):
            cls.__dispatch = {}
            for resultclass in resultclasses:
                cls.__dispatch[resultclass] = {}
                for (k, v) in resultclass.__dict__.items():
                    # All __promise__ return the same wrapper method, but they
                    # also do setup, inserting the method into the dispatch
                    # dict.
                    meth = cls.__promise__(resultclass, k, v)
                    if hasattr(cls, k):
                        continue
                    setattr(cls, k, meth)
            cls._delegate_str = str in resultclasses
            cls._delegate_unicode = six.text_type in resultclasses

            assert not (
                cls._delegate_str and cls._delegate_unicode
            ), "Cannot call lazy() with both str and unicode return types."

            if cls._delegate_unicode:
                cls.__unicode__ = cls.__unicode_cast
            elif cls._delegate_str:
                cls.__str__ = cls.__str_cast
        __prepare_class__ = classmethod(__prepare_class__)

        def __promise__(cls, klass, funcname, func):
            # Builds a wrapper around some magic method and registers that magic
            # method for the given type and method name.
            def __wrapper__(self, *args, **kw):
                # Automatically triggers the evaluation of a lazy value and
                # applies the given magic method of the result type.
                res = self.__func(*self.__args, **self.__kw)
                for t in type(res).mro():
                    if t in self.__dispatch:
                        return self.__dispatch[t][funcname](res, *args, **kw)
                raise TypeError("Lazy object returned unexpected type.")

            if klass not in cls.__dispatch:
                cls.__dispatch[klass] = {}
            cls.__dispatch[klass][funcname] = func
            return __wrapper__
        __promise__ = classmethod(__promise__)

        def __unicode_cast(self):
            return self.__func(*self.__args, **self.__kw)

        def __str_cast(self):
            return str(self.__func(*self.__args, **self.__kw))

        def __cmp__(self, rhs):
            if self._delegate_str:
                s = str(self.__func(*self.__args, **self.__kw))
            elif self._delegate_unicode:
                s = six.text_type(self.__func(*self.__args, **self.__kw))
            else:
                s = self.__func(*self.__args, **self.__kw)
            return cmp(s, rhs)  # noqa

        def __mod__(self, rhs):
            if self._delegate_str:
                return str(self) % rhs
            elif self._delegate_unicode:
                return six.text_type(self) % rhs
            else:
                raise AssertionError('__mod__ not supported for non-string types')

        def __deepcopy__(self, memo):
            # Instances of this class are effectively immutable. It's just a
            # collection of functions. So we don't need to do anything
            # complicated for copying.
            memo[id(self)] = self
            return self

    @wraps(func)
    def __wrapper__(*args, **kw):
        # Creates the proxy object, instead of the actual value.
        return __proxy__(args, kw)

    return __wrapper__


class Cache(dict):
    """Cache that stores object for the desired period of time"""
    def __init__(self, cachePeriod=None, timefun=None):
        self.__cachePeriod = cachePeriod
        if cachePeriod is not None and timefun is None:
            # Can't make this import on toplevel because of circular import
            from .sys.gettime import monoTime
            timefun = monoTime
        self.__timefun = timefun
        super(Cache, self).__init__()

    def __check(self, key):
        if self.__cachePeriod is not None:
            val = super(Cache, self).get(key)
            if val is not None:
                if self.__timefun() - val[0] > self.__cachePeriod:
                    del self[key]

    def __getitem__(self, key):
        self.__check(key)
        if self.__cachePeriod is None:
            return super(Cache, self).__getitem__(key)
        val = super(Cache, self).__getitem__(key)
        if val is None:
            raise KeyError(val)
        return val[1]

    def __setitem__(self, key, value):
        if self.__cachePeriod is None:
            return super(Cache, self).__setitem__(key, value)
        return super(Cache, self).__setitem__(key, (self.__timefun(), value))

    def __contains__(self, key):
        self.__check(key)
        return super(Cache, self).__contains__(key)

    def get(self, key, d=None):
        self.__check(key)
        if self.__cachePeriod is None:
            return super(Cache, self).get(key, d)
        val = super(Cache, self).get(key, d)

        return val if val is None else val[1]

    def __iter__(self):
        raise NotImplementedError

    def has_key(self, key):
        raise NotImplementedError

    def copy(self):
        raise NotImplementedError

    def items(self):
        raise NotImplementedError

    def iteritems(self):
        raise NotImplementedError

    def viewitems(self):
        raise NotImplementedError

    def values(self):
        raise NotImplementedError

    def itervalues(self):
        raise NotImplementedError

    def viewvalues(self):
        raise NotImplementedError

    def keys(self):
        raise NotImplementedError

    def iterkeys(self):
        raise NotImplementedError

    def viewkeys(self):
        raise NotImplementedError

    def pop(self, key, d=None):
        raise NotImplementedError

    def setdefault(self, key, d=None):
        raise NotImplementedError

    def update(self, *args, **kwargs):
        raise NotImplementedError

    @classmethod
    def fromkeys(cls, *args, **kwargs):
        raise NotImplementedError


class memoized(object):
    """
    Decorator that caches a function's return value each time it is called.
    If called later with the same arguments, the cached value is returned, and
    not re-evaluated.
    """

    def __init__(self, func, cachePeriod=None):
        self.func = func
        self.cachePeriod = cachePeriod
        self.cache = Cache(cachePeriod)
        self.methods = weakref.WeakKeyDictionary()

    def __call__(self, *args, **kwargs):
        if kwargs:
            cargs = (args, tuple(sorted(kwargs.items())))
        else:
            cargs = (args, ())
        try:
            return self.cache[cargs]
        except KeyError:
            value = self.func(*args, **kwargs)
            self.cache[cargs] = value
            return value
        except TypeError:
            # uncachable -- for instance, passing a list as an argument.
            # Better to not cache than to blow up entirely.
            return self.func(*args, **kwargs)

    def __repr__(self):
        """Return the function's docstring."""
        return self.func.__doc__

    def __get__(self, obj, objtype):
        """Support instance methods."""
        if obj is None:
            # it's access through class attribute
            return self
        cache = self.methods.setdefault(obj, Cache(self.cachePeriod))
        return MemoizedMethod(obj, self.func, cache)


class MemoizedMethod(object):
    """
    Decorator that caches instance method's results.
    In contrary with plain memoized it stores weakref to the object
    and prevents it from leaking.
    """

    def __init__(self, obj, func, cache):
        self.obj = obj
        self.func = func
        self.cache = cache

    def __call__(self, *args, **kwargs):
        if kwargs:
            cargs = (args, tuple(sorted(kwargs.items())))
        else:
            cargs = (args, ())

        try:
            return self.cache[cargs]
        except KeyError:
            value = self.func(self.obj, *args, **kwargs)
            self.cache[cargs] = value
            return value
        except TypeError:
            # uncachable -- for instance, passing a list as an argument.
            # Better to not cache than to blow up entirely.
            return self.func(self.obj, *args, **kwargs)


def cached(cachePeriod):
    def _cached(func):
        return memoized(func, cachePeriod)
    return _cached


class threadsafe(object):
    """
    Decorator which puts threading.Lock around function call.
    """
    def __init__(self, func):
        import threading
        self.lock = threading.Lock()
        self.func = func

    def __call__(self, *args, **kwargs):
        with self.lock:
            return self.func(*args, **kwargs)

    def __repr__(self):
        """
        Return the function's docstring.
        """
        return self.func.__doc__

    def __get__(self, obj, objtype):
        """
        Support instance methods.

        E.g. you can use:
        >>> @threadsafe
        >>> class SomeClass(object):
        ...     def someMethod(self):
        ...         pass

        And all class methods will be threadsafe :). Even class creation will be
        threadsafe (because __init__ method will :)).
        """

        return partial(self.__call__, obj)


def singleton(func):
    return memoized(threadsafe(memoized(func)))


def reliable(maxAttempts=3, exceptions=tuple(), timeout=None, sleep=time.sleep):
    """
    Decorator which try to call function expecting passed exceptions for at most attempts times making sleeps for
    timeout between attempts.
    @param maxAttempts: number of attempts to perform
    @param exceptions: topple of exceptions on which retry should be performed
    @param timeout: timeout between attempts
    """
    def decorator(func):

        @wraps(func)
        def decorated(*args, **kwargs):
            attempts = maxAttempts
            while True:
                try:
                    return func(*args, **kwargs)
                except exceptions:
                    attempts -= 1
                    if attempts >= 0:
                        if timeout is not None:
                            sleep(timeout)
                        continue
                    else:
                        raise
        return decorated
    return decorator


def Reliable(maxAttempts=3, exceptions=tuple(), timeout=None):
    def decorator(func):
        return reliable(maxAttempts, exceptions, timeout)()
    return decorator


class CachedState(object):
    __slots__ = ['__obj', '__cb', '__update_time', '__cache_period', '__timefun', '__doc__', '__call__', '__iter__']

    def __init__(self, obj, cb, cachePeriod=None, timefun=None):
        """
        The idea of the class is to wrap an object with
        the state that can expire, and trigger the state
        update callback if needed on each access.

        :param obj: wrapped object
        :param cb: callable to trigger on object expiration
        :type cb: callable or str. If str, the callable will be
                  retrieved by `getattr(obj, cb)`
        :param cachePeriod: period before state update in seconds.
                            If None, cached state will never be updated.
                            Note, that newly wrapped object is always
                            considered as not cached and will be updated
                            on the first access.
        :param callable timefun: function to retrieve the time. By default
                                 :func:`~kernel.util.sys.gettime.monoTime`
                                 is used.
        """
        self.__obj = obj
        self.__cb = cb
        self.__update_time = None
        self.__cache_period = cachePeriod
        if cachePeriod is not None and timefun is None:
            from .sys.gettime import monoTime
            timefun = monoTime
        self.__timefun = timefun
        self.__doc__ = self.__obj.__doc__
        if hasattr(obj, '__call__'):
            object.__setattr__(self, '__call__', partial(self.__call, '__call__'))
        if hasattr(obj, '__iter__'):
            object.__setattr__(self, '__iter__', partial(self.__call, '__iter__'))

    def __update_state__(self):
        if (
            self.__update_time is None
            or (
                self.__cache_period is not None
                and self.__timefun() - self.__update_time >= self.__cache_period
            )
        ):
            time = self.__timefun()
            cb = getattr(self.__obj, self.__cb) if isinstance(self.__cb, six.string_types) else self.__cb
            cb()
            self.__update_time = time

    def __call(self, name, *args, **kwargs):
        self.__update_state__()
        return getattr(self.__obj, name)(*args, **kwargs)

    def __getattr__(self, key):
        if key.startswith('_CachedState__'):
            return object.__getattr__(self, key)
        else:
            self.__update_state__()
            return getattr(self.__obj, key)

    def __setattr__(self, key, val):
        if key.startswith('_CachedState__'):
            return object.__setattr__(self, key, val)
        else:
            self.__update_state__()
            return setattr(self.__obj, key, val)

    def __delattr__(self, key):
        if key.startswith('_CachedState__'):
            return object.__delattr__(self, key)
        else:
            self.__update_state__()
            return delattr(self.__obj, key)

    def __str__(self):
        self.__update_state__()
        return str(self.__obj)

    def __repr__(self):
        self.__update_state__()
        return repr(self.__obj)
