"""
Descendants of gevent.event
"""

import weakref

import gevent.event
import gevent.greenlet

from .linkable import deduceReceiver


class LinkedSet(Exception):
    """Raised when a linked event is set"""
    msg = "{0!r} set"

    def __init__(self, event):
        super(LinkedSet, self).__init__(self.msg.format(event), event)


class GreenletEventLink(object):
    """
    As above
    """
    __slots__ = ['greenlet']

    def __init__(self, greenlet):
        self.greenlet = greenlet

    def __call__(self, event):
        self.greenlet.throw(LinkedSet(event))

    def __hash__(self):
        return hash(self.greenlet)

    def __eq__(self, other):
        return self.greenlet == getattr(other, 'greenlet', other)

    def __str__(self):
        return str(self.greenlet)

    def __repr__(self):
        return repr(self.greenlet)


class LinkMixIn(object):
    """
    Our mixIn for Event and AsyncResult which adds link method
    """
    def ready(self):
        raise NotImplementedError

    def rawlink(self, _):
        raise NotImplementedError

    def link(self, receiver=None, GreenletLink=gevent.greenlet.GreenletLink, SpawnedLink=gevent.greenlet.SpawnedLink):
        """
        As above
        """
        current = gevent.greenlet.getcurrent()
        receiver = deduceReceiver(receiver, GreenletLink, SpawnedLink)
        if self.ready() and receiver is current:
            receiver(self)
        else:
            self.rawlink(receiver)


class UserDataMixIn(object):
    def __init__(self, *_, **__):
        del self.userData

    @property
    def userData(self):
        if isinstance(self.__userData, weakref.ref):
            return self.__userData()
        else:
            return self.__userData

    @userData.setter
    def userData(self, userData=None):
        """
        :param userData: any user data
        """
        try:
            self.__userData = weakref.ref(userData)
        except TypeError:
            self.__userData = userData

    @userData.deleter
    def userData(self):
        self.__userData = None


class Event(gevent.event.Event, LinkMixIn, UserDataMixIn):
    """
    Our descendant of Event with link method and userdata
    """
    def link(self, receiver=None, GreenletLink=GreenletEventLink, SpawnedLink=gevent.greenlet.SpawnedLink):
        LinkMixIn.link(self, receiver, GreenletLink, SpawnedLink)

    def unlink(self, receiver=None):
        if receiver is None:
            receiver = gevent.getcurrent()
        gevent.event.Event.unlink(self, receiver)


class AsyncResult(gevent.event.AsyncResult, LinkMixIn, UserDataMixIn):
    """
    Our descendant of AsyncResult with link method
    """
    def link_value(
        self,
        receiver=None,
        GreenletLink=gevent.greenlet.SuccessGreenletLink,
        SpawnedLink=gevent.greenlet.SuccessSpawnedLink
    ):
        """Like :meth:`link` but *receiver* is only notified when the greenlet has completed successfully"""
        self.link(receiver=receiver, GreenletLink=GreenletLink, SpawnedLink=SpawnedLink)

    def link_exception(
            self,
            receiver=None,
            GreenletLink=gevent.greenlet.FailureGreenletLink,
            SpawnedLink=gevent.greenlet.FailureSpawnedLink
    ):
        """Like :meth:`link` but *receiver* is only notified when the greenlet dies because of unhandled exception"""
        self.link(receiver=receiver, GreenletLink=GreenletLink, SpawnedLink=SpawnedLink)

    def unlink(self, receiver=None):
        if receiver is None:
            receiver = gevent.getcurrent()
        gevent.event.AsyncResult.unlink(self, receiver)
