import sys
import traceback

import gevent
import gevent.core


def deduceReceiver(
        receiver=None,
        GreenletLink=gevent.greenlet.GreenletLink,
        SpawnedLink=gevent.greenlet.SpawnedLink
):
    current = gevent.greenlet.getcurrent()
    if receiver is None or receiver is current:
        return GreenletLink(current)
    elif not callable(receiver):
        if isinstance(receiver, gevent.greenlet.Greenlet):
            return GreenletLink(receiver)
        else:
            raise TypeError('Expected callable or greenlet: %r' % (receiver, ))
    else:
        return SpawnedLink(receiver)


class Linkable(object):
    __slots__ = [
        '__links',
    ]

    def __init__(self):
        self.__links = set()

    def rawlink(self, receiver):
        self.__links.add(receiver)

    def rawunlink(self, receiver):
        self.__links.discard(receiver)

    def link(self, receiver=None, GreenletLink=gevent.greenlet.GreenletLink, SpawnedLink=gevent.greenlet.SpawnedLink):
        self.rawlink(deduceReceiver(receiver, GreenletLink, SpawnedLink))

    def unlink(self, receiver=None, GreenletLink=gevent.greenlet.GreenletLink, SpawnedLink=gevent.greenlet.SpawnedLink):
        self.rawunlink(deduceReceiver((receiver, GreenletLink, SpawnedLink)))

    def _clearLinks(self):
        self.__links = set()

    def _notifyLinks(self):
        gevent.core.active_event(self.__doNotifyLinks, self.__links.copy())

    def _notifyLinksOnce(self):
        gevent.core.active_event(self.__doNotifyLinksOnce, self.__links)
        self.__links = set()

    def __doNotifyLink(self, link):
        try:
            link(self)
        except:
            traceback.print_exc()
            try:
                sys.stderr.write('Failed to notify link %r of %r\n\n' % (link, self))
            except:
                traceback.print_exc()

    def __doNotifyLinks(self, links):
        for link in links:
            if link in self.__links:
                self.__doNotifyLink(link)

    def __doNotifyLinksOnce(self, links):
        for link in links:
            self.__doNotifyLink(link)
