"""
Queue and PriorityQueue
"""

from collections import deque
from heapq import heappush, heappop
from itertools import chain

import gevent
import gevent.pool
import gevent.event


class Queue(gevent.pool.Group):
    """
    Ordered pool
    """
    def __init__(self, size=1, greenlet_class=None):
        super(Queue, self).__init__()
        self.__size = size
        if greenlet_class is not None:
            self.greenlet_class = greenlet_class
        self.__available_event = gevent.event.Event()
        self.__available_event.set()
        self.__queue = deque()

    def __iter__(self):
        """
        :return: iterable over all lets
        """
        return chain(super(Queue, self).__iter__(), self.__queue)

    def iterRunning(self):
        """
        :return: iterable over currently running lets
        """
        return super(Queue, self).__iter__()

    def iterQueued(self):
        """
        :return: iterable over queued lets
        """
        return iter(self.__queue)

    def add(self, greenlet):
        if len(self) < self.__size:
            super(Queue, self).add(greenlet)
            greenlet.start()
        else:
            self.__available_event.clear()
            self.__queue.append(greenlet)

    def discard(self, greenlet):
        super(Queue, self).discard(greenlet)
        while len(self) < self.__size and self.__queue:
            greenlet = self.__queue.popleft()
            super(Queue, self).add(greenlet)
            greenlet.start()
        if len(self) < self.__size:
            self.__available_event.set()

    def full(self):
        return len(self) >= self.__size

    def wait_available(self):
        self.__available_event.wait()

    def kill(self, exception=gevent.GreenletExit, block=True, timeout=None):
        while self.__queue:
            greenlet = self.__queue.pop()
            greenlet.kill(exception, block, timeout)
        super(Queue, self).kill(exception, block, timeout)

    def killone(self, greenlet, exception=gevent.GreenletExit, block=True, timeout=None):
        try:
            self.__queue.remove(greenlet)
        except ValueError:
            super(Queue, self).killone(greenlet, exception, block, timeout)
        else:
            greenlet.kill(exception, block, timeout)

    def start(self, greenlet):
        self.add(greenlet)

    def spawn(self, *args, **kwargs):
        greenlet = self.greenlet_class(*args, **kwargs)
        self.add(greenlet)
        return greenlet

    def spawn_link(self, *args, **kwargs):
        greenlet = self.greenlet_class(*args, **kwargs)
        self.add(greenlet)
        greenlet.link()
        return greenlet

    def spawn_link_value(self, *args, **kwargs):
        greenlet = self.greenlet_class(*args, **kwargs)
        self.add(greenlet)
        greenlet.link_value()
        return greenlet

    def spawn_link_exception(self, *args, **kwargs):
        greenlet = self.greenlet_class(*args, **kwargs)
        self.add(greenlet)
        greenlet.link_exception()
        return greenlet


# noinspection PyMethodOverriding
class PriorityQueue(gevent.pool.Group):
    """
    Ordered pool with priorities
    """
    def __init__(self, size=1, greenlet_class=None):
        super(PriorityQueue, self).__init__()
        self.__size = size
        if greenlet_class is not None:
            self.greenlet_class = greenlet_class
        self.__available_event = gevent.event.Event()
        self.__available_event.set()
        self.__queue = []
        self.__counter = 0

    def __iter__(self):
        """
        :return: iterable over all lets
        """
        return chain(super(PriorityQueue, self).__iter__(), self.__queue)

    def iterRunning(self):
        """
        :return: iterable over currently running lets
        """
        return super(PriorityQueue, self).__iter__()

    def iterQueued(self):
        """
        :return: iterable over queued lets
        """
        return iter(self.__queue)

    def add(self, greenlet, priority=0):
        if len(self) < self.__size:
            super(PriorityQueue, self).add(greenlet)
            greenlet.start()
        else:
            self.__available_event.clear()
            heappush(self.__queue, (priority, self.__counter, greenlet))
            self.__counter += 1

    def discard(self, greenlet):
        super(PriorityQueue, self).discard(greenlet)
        while len(self) < self.__size and self.__queue:
            _, _, greenlet = heappop(self.__queue)
            super(PriorityQueue, self).add(greenlet)
            greenlet.start()
        if len(self) < self.__size:
            self.__available_event.set()

    def full(self):
        return len(self) >= self.__size

    def wait_available(self):
        self.__available_event.wait()

    def kill(self, exception=gevent.GreenletExit, block=True, timeout=None):
        while self.__queue:
            _, _, greenlet = self.__queue.pop()
            greenlet.kill(exception, block, timeout)
        super(PriorityQueue, self).kill(exception, block, timeout)

    def killone(self, greenlet, exception=gevent.GreenletExit, block=True, timeout=None):
        try:
            self.__queue.remove(greenlet)
        except ValueError:
            super(PriorityQueue, self).killone(greenlet, exception, block, timeout)
        else:
            greenlet.kill(exception, block, timeout)

    def start(self, greenlet, priority=0):
        self.add(greenlet, priority)

    def spawn(self, priority=0, *args, **kwargs):
        greenlet = self.greenlet_class(*args, **kwargs)
        self.add(greenlet, priority)
        return greenlet

    def spawn_link(self, priority=0, *args, **kwargs):
        greenlet = self.greenlet_class(*args, **kwargs)
        self.add(greenlet, priority)
        greenlet.link()
        return greenlet

    def spawn_link_value(self, priority=0, *args, **kwargs):
        greenlet = self.greenlet_class(*args, **kwargs)
        self.add(greenlet, priority)
        greenlet.link_value()
        return greenlet

    def spawn_link_exception(self, priority=0, *args, **kwargs):
        greenlet = self.greenlet_class(*args, **kwargs)
        self.add(greenlet, priority)
        greenlet.link_exception()
        return greenlet
