"""
Asynchronous primitives to be used in threading
"""

__author__ = 'torkve'

import os
import gevent.socket

from ..net.pipe import Pipe


class AsyncResult(object):
    __unknownState__ = object()
    __normalResult__ = object()
    __exceptionResult__ = object()

    waitDelay = 0.01

    def __init__(self):
        super(AsyncResult, self).__init__()
        self.__eventPipe = Pipe()
        self._init()

    def _init(self):
        self.__eventPipe.open()
        self.__result = self.__unknownState__
        self.__resultType = None

    def clear(self):
        self.__eventPipe.close()
        self._init()

    def set(self, value):
        self.__result = value
        self.__resultType = self.__normalResult__
        self.__eventPipe.writeAll('1')
        self.__eventPipe.closeW()

    def setException(self, value):
        self.__result = value
        self.__resultType = self.__exceptionResult__
        self.__eventPipe.writeAll('1')
        self.__eventPipe.closeW()

    set_exception = setException

    def _returnResult(self):
        if self.__result is self.__unknownState__:
            raise AsyncResult.Timeout()
        elif self.__resultType is self.__exceptionResult__:
            raise self.__result
        else:
            return self.__result

    def _wait(self, timeout=None):
        if timeout is not None and timeout < 0:
            timeout = 0
        r = self.__eventPipe.r
        try:
            gevent.socket.wait_read(r, timeout=timeout, timeout_exc=AsyncResult.Timeout)
            os.read(r, 1)
            return True
        except AsyncResult.Timeout:
            return False

    def get(self, timeout=None):
        if self.__result is self.__unknownState__:
            if self._wait(timeout):
                self.__eventPipe.closeR()
        return self._returnResult()

    wait = get

    def is_set(self):
        return self.__result is not self.__unknownState__

    isSet = is_set

    def __del__(self):
        self.__eventPipe.close()

    class Timeout(Exception):
        pass
