# Deprecated
import six
import time
import traceback
import types
from threading import RLock

from ..console import colors

__all__ = [
    "Log"
]


def fullFormat(logLevel, desc, logString):
    sec = time.time()
    ms = int((sec - int(sec)) * 1000)
    st = u"" + time.strftime("%d/%m %H:%M:%S", time.localtime(sec)) + \
         " +%03d " % ms + logLevel + " [ " + desc.decode("utf-8") + " ] "
    st += logString.decode("utf-8")
    return st


def plainFormat(logLevel, desc, logString):
    return u"" + logLevel + " [ " + desc + " ] " + logString.decode("utf-8")


def fullPlainFormat(logLevel, desc, logString):  # @UnusedVariable
    return u"" + logString.decode("utf-8")


class Log:
    writers = {}
    lock = RLock()  # in fact writers lock

    prefix = ''

    class Levels:
        proto = "PROTO"
        debug = "DEBUG"
        info = "INFO "
        warn = "WARN "
        error = "ERROR"
        fatal = "FATAL"
        trace = "TRACE"

    @classmethod
    def clear(cls):
        with cls.lock:
            for writer in cls.writers.keys():
                try:
                    writer.close()
                except IOError:
                    pass
            cls.writers = {}

    @classmethod
    def addWriter(cls, writer, formats=None):
        with cls.lock:
            if not formats:
                formats = cls.defaultLogFormats
            cls.writers[writer] = formats

    @classmethod
    def removeWriter(cls, writer):
        with cls.lock:
            if writer in cls.writers:
                cls.writers.pop(writer)

    @classmethod
    def protoEnabled(cls):
        with cls.lock:
            proto = cls.Levels.proto
            for format in cls.writers.itervalues():
                if proto in format:
                    return True
            return False

    def __init__(self, desc=None, object=None):
        if isinstance(desc, six.string_types):
            self._desc = desc
            self.object = object
        else:
            self._desc = None
            self.object = desc if desc else object

        self.disabledLevels = set()

    def __call__(self, logString, data=None):
        self.info(logString, data)

    # TODO: rename to downgradeLevel
    def disableLevel(self, level):
        self.disabledLevels.add(level)

    def enableLevel(self, level):
        if level in self.disabledLevels:
            self.disabledLevels.remove(level)

    disable = lambda self: self.enable(False)

    def desc(self):
        if self.object:
            return str(self.object)
        else:
            return str(self._desc)

    def info(self, logString, data=None):
        self._log(self.Levels.info, logString, data)

    def warn(self, logString, data=None):
        self._log(self.Levels.warn, logString, data)

    def error(self, logString, data=None):
        self._log(self.Levels.error, logString, data)

    def debug(self, logString, data=None):
        self._log(self.Levels.debug, logString, data)

    def proto(self, logString, data=None):
        self._log(self.Levels.proto, logString, data)

    def fatal(self, logString, data=None):
        self._log(self.Levels.fatal, logString, data)
        self.trace()

    def trace(self):
        traceback.print_exc(file=self)

    def write(self, logString):
        self._log(self.Levels.trace, logString)

    def flush(self, logString):
        # Auto flush after each write
        pass

    def _logToWriter(self, writer, formats, logLevel, logString, data=None):
        if logLevel in self.disabledLevels:
            logLevel = "DEBUG"
        if data:
            data = (u"" + data.decode("utf-8")).encode("utf-8")
            if len(data.split("\n")) <= 1:
                logString = " ".join((logString, data))
                data = None
        if logLevel in formats:
            color = None
            formatter = formats[logLevel]
            if isinstance(formatter, types.TupleType):
                formatter, color = formatter
            if color:
                logLevel = color + logLevel + colors.default
            if self.prefix:
                logString = self.prefix + logString
            writer.write((formatter(logLevel, self.desc(), logString) + "\n").encode("utf-8"))
            if data:
                if not data.endswith("\n"):
                    data += "\n"
                writer.write(data)
            writer.flush()

    def _log(self, logLevel, logString, data=None):
        with self.lock:
            for writer, formats in self.writers.items():
                if writer != self.object:  # Prevent echo
                    self._logToWriter(writer, formats, logLevel, logString, data)

    defaultLogFormats = {
        Levels.debug: fullFormat,
        Levels.proto: fullFormat,
        Levels.info: fullFormat,
        Levels.warn: fullFormat,
        Levels.error: fullFormat,
        Levels.fatal: fullFormat,
        Levels.trace: plainFormat
    }

    colorLogFormat = {
        Levels.debug: (fullFormat, colors.gray),
        Levels.proto: (fullFormat, colors.gray),
        Levels.info: fullFormat,
        Levels.warn: (fullFormat, colors.yellow),
        Levels.error: (fullFormat, colors.magenta),
        Levels.fatal: (fullFormat, colors.magenta + colors.bold),
        Levels.trace: plainFormat,
    }
