from __future__ import absolute_import, print_function, division

import six
import sys
import time
import logging

from logging import *  # noqa
from logging import handlers  # noqa

from .console import isTerminal
from .errors import formatException


# Patch logging to have getChild function from Python 2.7.x
if not hasattr(logging.Logger, 'getChild'):
    def _getChild(self, suffix):
        if self.root is not self:
            suffix = '.'.join((self.name, suffix))
        return self.manager.getLogger(suffix)

    logging.Logger.getChild = _getChild


# Patch logging to have captureWarnings() functionality from Python 2.7.x
if not hasattr(logging, 'captureWarnings'):
    logging._warnings_showwarning = None

    def _showwarning(message, category, filename, lineno, file=None, line=None):
        import logging
        import warnings

        if file is not None:
            if logging._warnings_showwarning is not None:
                logging._warnings_showwarning(message, category, filename, lineno, file, line)
        else:
            s = warnings.formatwarning(message, category, filename, lineno, line)
            logger = logging.getLogger('py.warnings')
            logger.warning('%s', s)

    logging._showwarning = _showwarning

    def captureWarnings(capture):
        # Need import again, since old imports are not usable after we replaced
        # package in sys.modules
        import logging
        import warnings

        if capture:
            if logging._warnings_showwarning is None:
                logging._warnings_showwarning = warnings.showwarning
                warnings.showwarning = logging._showwarning
        else:
            if logging._warnings_showwarning is not None:
                warnings.showwarning = logging._warnings_showwarning
                logging._warnings_showwarning = None

    logging.captureWarnings = captureWarnings


# Patch logging to have short level names
_levelNameAliases = {
    # LEVEL:    (1char, 3 chars, 4 chars, full)
    'PROTOCOL': ('P',   'PRT',   'PROT',  'PROTOCOL'),
    'DEBUG':    ('D',   'DBG',   'DEBG',  'DEBUG'),
    'INFO':     ('I',   'INF',   'INFO',  'NORMAL'),
    'WARNING':  ('W',   'WRN',   'WARN',  'WARNING'),
    'ERROR':    ('E',   'ERR',   'ERRR',  'ERROR'),
    'CRITICAL': ('C',   'CRI',   'CRIT',  'CRITICAL'),
    'NOTSET':   ('-',   '---',   '----',  'NOTSET'),
}

protocolLevel = 9
logging.addLevelName(protocolLevel, 'PROTOCOL')

if not hasattr(logging.Logger, 'proto'):
    # Add shortcut to protocol log level
    def proto(self, msg, *args, **kwargs):
        """
        Log on protocol log level
        :param msg: Message to log
        """
        self.log(protocolLevel, msg, *args, **kwargs)

    logging.Logger.proto = proto

if not hasattr(logging.Logger, 'normal'):
    # Compatibility with old skynet logging level
    logging.Logger.normal = logging.Logger.info


def renameLevels(chars=None):
    assert chars in (None, 1, 3, 4), 'We only support unlimited and 1-3-4 char themes'

    if chars == 1:
        idx = 0
    elif chars == 3:
        idx = 1
    elif chars == 4:
        idx = 2
    else:
        idx = 3

    for level, aliases in list(_levelNameAliases.items()):
        selectedAlias = aliases[idx]

        levelno = logging.getLevelName(level)
        logging.addLevelName(levelno, selectedAlias)


logging._levelNameAliases = _levelNameAliases
logging.renameLevels = renameLevels


class ExtendedFormatter(logging.Formatter):
    def __init__(
        self,
        fmt='%(asctime)s - %(levelname)s - %(name)s - %(message)s',
        fmtName='%(name)s',
        fmtLevelno='%(levelno)d',
        fmtLevelname='%(levelname)s',
        fmtPathname='%(pathname)s',
        fmtFilename='%(filename)s',
        fmtModule='%(module)s',
        fmtLineno='%(lineno)d',
        fmtFuncName='%(funcName)s',
        fmtCreated='%(created)f',
        fmtAsctime='%(datetime)s,%(msecs)03d',
        fmtDatetime='%Y-%m-%d %H:%M:%S',
        fmtMsecs='%(msecs)d',
        fmtRelativeCreated='%(relativeCreated)d',
        fmtThread='%(thread)d',
        fmtThreadName='%(threadName)s',
        fmtProcess='%(process)d',
        fmtProcessName='%(processName)s',
        fmtHost='%(host)s',
        fmtMessage='%(message)s',
    ):
        self.__dict__.update(locals())

    def formatTime(self, record, datefmt=None):
        ct = self.converter(record.created)
        datetime = time.strftime(datefmt if datefmt is not None else self.fmtDatetime, ct)
        return self.fmtAsctime % {
            'datetime': datetime,
            'msecs': record.msecs
        }

    def formatSimple(self, record, type_):
        default = '%%(%s)s' % (type_, )
        return getattr(self, 'fmt%s' % (type_[0].upper() + type_[1:], ), default) % {
            type_: getattr(record, type_)
        }

    formatName = lambda self, record: self.formatSimple(record, 'name')
    formatLevelno = lambda self, record: self.formatSimple(record, 'levelno')
    formatLevelname = lambda self, record: self.formatSimple(record, 'levelname')
    formatPathname = lambda self, record: self.formatSimple(record, 'pathname')
    formatFilename = lambda self, record: self.formatSimple(record, 'filename')
    formatModule = lambda self, record: self.formatSimple(record, 'module')
    formatLineno = lambda self, record: self.formatSimple(record, 'lineno')
    formatFuncName = lambda self, record: self.formatSimple(record, 'funcName')
    formatCreated = lambda self, record: self.formatSimple(record, 'created')
    formatMsecs = lambda self, record: self.formatSimple(record, 'msecs')
    formatRelativeCreated = lambda self, record: self.formatSimple(record, 'relativeCreated')
    formatThread = lambda self, record: self.formatSimple(record, 'thread')
    formatThreadName = lambda self, record: self.formatSimple(record, 'threadName')
    formatProcess = lambda self, record: self.formatSimple(record, 'process')
    formatProcessName = lambda self, record: self.formatSimple(record, 'processName')
    formatHost = lambda self, record: self.formatSimple(record, 'host')
    formatMessage = lambda self, record: self.formatSimple(record, 'message')

    def prepare(self, record):
        record.message = record.getMessage()
        if '%(asctime)' in self.fmt:
            record.asctime = self.formatTime(record)
        if not hasattr(record, 'host'):  # Beware: hack for local and remote log records mix
            record.host = 'localhost'

        if record.exc_text:
            if not isinstance(record.message, six.string_types):
                record.message = str(record.message)
            if not record.message.endswith('\n'):
                record.message += '\n'
            try:
                record.message += record.exc_text
            except UnicodeError:
                # Sometimes filenames have non-ASCII chars, which can lead
                # to errors when s is Unicode and record.exc_text is str
                # See issue 8924
                record.message += record.exc_text.decode(sys.getfilesystemencoding())

        fmtData = {'asctime': record.asctime}

        for simpleItem in list(record.__dict__.keys()):
            if simpleItem in fmtData:
                continue
            if '%%(%s)' % simpleItem in self.fmt:
                try:
                    formatter = getattr(self, 'format%s' % (simpleItem[0].upper() + simpleItem[1:]))
                except AttributeError:
                    formatter = lambda record: self.formatSimple(record, simpleItem)
                    setattr(self, 'format%s' % (simpleItem[0].upper() + simpleItem[1:]), formatter)
                fmtData[simpleItem] = formatter(record)

        return fmtData

    def format(self, record):
        fmtData = self.prepare(record)
        return self.fmt % fmtData

    def formatException(self, excInfo):
        return formatException(excInfo=excInfo)


class ColoredFormatter(ExtendedFormatter):
    COLORS = dict(zip(
        ('BLACK', 'RED', 'GREEN', 'YELLOW', 'BLUE', 'MAGENTA', 'CYAN', 'WHITE'),
        [{'num': x} for x in range(30, 38)]
    ))

    ALIASES = {
        'NORMAL': 'INFO',
        'UNKNOWN': 'NOTSET'
    }

    for key, values in six.iteritems(_levelNameAliases):
        for value in values:
            ALIASES[value] = key

    for color, info in list(COLORS.items()):
        COLORS[color]['normal'] = '\033[0;%dm' % info['num']
        COLORS[color]['bold'] = '\033[1;%dm' % info['num']

    RESET_SEQ = '\033[m'

    def __init__(self, **kwargs):
        # Defaults for colorMap and Hungry levels
        colorMap = {
            'levels': {
                # LEVEL:    (COLOR,    BOLD_OR_NOT)
                'PROTOCOL': ('WHITE',  True),
                'DEBUG':    ('BLACK',  True),
                'INFO':     ('GREEN',  False),
                'WARNING':  ('YELLOW', True),
                'ERROR':    ('RED',    False),
                'CRITICAL': ('RED',    True),
            },
            'parts': {
                # PARTNAME: (COLOR,    BOLD_OR_NOT
                'asctime':  ('BLACK',  True),
                'name':     ('CYAN',   True),
                'host':     ('BLACK',  True),
            },
        }

        hungryLevels = {
            'CRITICAL': ('full', ),             # Critical logs will have colorized full log message by default
            'ERROR': ('message', 'levelname'),  # Error logs will be colorized too (only message part)
        }

        self.useColors = kwargs.pop('useColors', True) and isTerminal()

        for key, value in list(kwargs.items()):
            dict_ = None
            if key.startswith('color'):
                dict_ = colorMap['parts']
                key2 = key[len('color'):].lower()
            elif key.startswith('level'):
                dict_ = colorMap['levels']
                key2 = key[len('level'):].upper()

            if dict_ is None:
                continue

            bold = value.endswith('+')
            value = value.rstrip('+')
            dict_[key2] = (value.upper(), bold)
            kwargs.pop(key)

        for levelName, childNames in kwargs.pop('hungryLevels', {}).items():
            hungryLevels[levelName.upper()] = childNames

        self.colorMap = colorMap
        self.hungryLevels = hungryLevels

        ExtendedFormatter.__init__(self, **kwargs)

    def colorize(self, string, info):
        """ Colorize string.

        :arg info: is a tuple with [color_name, bold_or_not] items.
        """

        return ''.join((
            self.COLORS[info[0]]['normal' if not info[1] else 'bold'],
            string,
            self.RESET_SEQ,
        ))

    def format(self, record):
        if record.exc_info:
            # Cache the traceback text to avoid converting it multiple times
            # (it's constant anyway)
            if not record.exc_text:
                record.exc_text = self.formatException(record.exc_info)

        if not self.useColors:
            return ExtendedFormatter.format(self, record)

        data = self.prepare(record)

        levelname = self.ALIASES.get(record.levelname, record.levelname)
        levelColor = self.colorMap['levels'].setdefault(levelname, None)
        if levelColor is not None:
            levelHungry = self.hungryLevels.setdefault(levelname, ('levelname', ))
            if 'full' in levelHungry:
                # Colorize full message and return immidiately if we are
                # very very hungry level
                return self.colorize(self.fmt % data, levelColor)
        else:
            levelHungry = ()

        # Colorize all specified parts
        for partName, partColor in list(self.colorMap['parts'].items()):
            if partName in levelHungry:
                partColor = levelColor
            if partName not in data:
                continue
            data[partName] = self.colorize(data[partName], partColor)

        # Colorize all hungry parts by level color
        if levelColor is not None:
            for partName in ['levelname'] + list(levelHungry):
                data[partName] = self.colorize(data[partName], levelColor)

        return self.fmt % data


class MessageAdapter(LoggerAdapter):  # noqa
    def __init__(self, logger, fmt='%(message)s', data=None):
        if data is None:
            data = {}
        else:
            assert 'message' not in data
        self.logger = logger
        self.fmt = fmt
        self.data = data

    def process(self, msg, kwargs):
        self.data['message'] = msg
        msg = self.fmt % self.data
        return msg, kwargs

    def getChild(self, suffix):
        data = self.data.copy()
        data.pop('message', None)
        return type(self)(self.logger.getChild(suffix), self.fmt, data)


__default__ = object()


def initialize(
    logger=__default__,
    level=__default__,
    levelChars=__default__,
    handler=__default__,
    formatter=__default__,
    appendHost=False
):

    if logger is __default__:
        logger = getLogger()  # noqa

    if level is __default__:
        logger.setLevel(DEBUG)  # noqa

    elif level is not None:
        logger.setLevel(level)

    if levelChars is __default__:
        renameLevels(1)
    elif levelChars is not None:
        renameLevels(levelChars)

    if handler is __default__:
        handler = logging.StreamHandler(sys.stderr)
        handler.setLevel(protocolLevel)

    assert handler is not None, 'Need handler'

    useColors = False
    if (
        isinstance(handler, logging.StreamHandler) and
        hasattr(handler.stream, 'fileno')
    ):
        try:
            fileno = handler.stream.fileno()
        except (AttributeError, ValueError):  # stream is not good file-object
            pass
        else:
            useColors = isTerminal(fileno)

    if appendHost:
        fmt = '%(asctime)s %(host)s %(levelname)s %(name)s  %(message)s'
    else:
        fmt = '%(asctime)s %(levelname)s %(name)s  %(message)s'

    if formatter is __default__:
        formatter = ColoredFormatter(
            useColors=useColors,
            fmt=fmt,
            fmtName='[%(name)-7s]',
            fmtLevelname='[%(levelname)-1s]',
            fmtAsctime='%(datetime)s.%(msecs)003d',
            hungryLevels={
                'info': ['message'],
                'debug': ['message'],
                'protocol': ['message'],
                'warning': ['message'],
                'error': ['message', 'name']
            }
        )

    if formatter:
        handler.setFormatter(formatter)

    logger.addHandler(handler)

    return logger
