import ctypes
import socket
import os

from ..sys import libc

__all__ = [
    "IfAddr",
    "getIfAddrs",
]

_reverseAF = {}

_osname = os.uname()[0].lower()


class Flags(object):
    IFF_UP = 0x1
    IFF_BROADCAST = 0x2
    IFF_DEBUG = 0x4
    IFF_LOOPBACK = 0x8
    IFF_POINTOPOINT = 0x10
    IFF_NOTRAILERS = 0x20
    IFF_SMART = 0x20
    IFF_RUNNING = 0x40
    IFF_DRV_RUNNING = 0x40
    IFF_NOARP = 0x80
    IFF_PROMISC = 0x100
    IFF_ALLMULTI = 0x200
    IFF_MASTER = 0x400
    IFF_DRV_OACTIVE = 0x400
    IFF_SLAVE = 0x800
    IFF_SIMPLEX = 0x800
    IFF_MULTICAST = 0x1000
    IFF_PORTSEL = 0x2000
    IFF_AUTOMEDIA = 0x4000
    IFF_DYNAMIC = 0x8000
    IFF_LOWER_UP = 0x10000
    IFF_DORMANT = 0x20000
    IFF_ECHO = 0x40000

for af in dir(socket):
    if af.startswith('AF_'):
        _reverseAF[getattr(socket, af)] = af


class IfAddr(object):
    __slots__ = [
        "name",
        "family",
        "addr",
        "netmask",
        "broadaddr",
        "flags",
        "scopeid",
        "list"
    ]

    def __repr__(self):
        return "<{0}[{1}]>".format(self.__class__.__name__, self)

    def __str__(self):
        return "{0} {1} {2}/{3}{4}".format(
            self.name,
            _reverseAF.get(self.family, self.family),
            self.addr,
            self.netmask,
            " scopeid={0}".format(self.scopeid) if self.scopeid else ""
        )

    def __init__(self, name, family, addr, netmask, broadaddr, flags, scopeid=0):
        self.name = name
        self.family = family
        self.addr = addr
        self.netmask = netmask
        self.broadaddr = broadaddr
        self.flags = flags
        self.scopeid = scopeid
        self.list = [family, addr, netmask, broadaddr, flags, scopeid]

    def __eq__(self, other):
        return type(self) is type(other) and self.name == other.name and self.list == other.list

    def __getitem__(self, item):
        return self.list[item]


def getIfAddrs():
    ifAddrs = ctypes.POINTER(__IfAddrs)()
    result = {}

    def lenFamily(sockAddr):
        family = sockAddr.lenFamily2
        len = sockAddr.lenFamily1
        if not family:
            family = len
            len = 16
            if family == socket.AF_INET6:
                len = 28
        return max(len, 16), family

    def getNameInfo(sockAddrP):
        if not sockAddrP:
            return

        buf = ctypes.create_string_buffer(socket.NI_MAXHOST)
        len, family = lenFamily(sockAddrP.contents)
        libc().getnameinfo(
            sockAddrP, len,
            ctypes.byref(buf), socket.NI_MAXHOST, 0, 0,
            socket.NI_NUMERICHOST
        )
        res = ctypes.cast(buf, ctypes.c_char_p).value
        if not res and family in (socket.AF_INET, socket.AF_INET6):
            inAddrP = ctypes.pointer(ctypes.cast(sockAddrP,
                                                 ctypes.POINTER(__SockAddrIn if family == socket.AF_INET else __SockAddrIn6)
                                                 ).contents.addr)

            libc().inet_ntop(family, inAddrP, ctypes.byref(buf), socket.NI_MAXHOST)
            res = ctypes.cast(buf, ctypes.c_char_p).value
#        pos = res.find("%")
#        if pos >= 0:
#            res = res[:pos]
        return res

    try:
        libc().getifaddrs(ctypes.byref(ifAddrs))
        ifAddrsC = ifAddrs
        while ifAddrsC:
            if ifAddrsC.contents.addr:
                addresses = result.setdefault(ifAddrsC.contents.name, [])
                family = lenFamily(ifAddrsC.contents.addr.contents)[1]
                scopeid = 0
                if (
                    # In the IPv6 addresses...
                    family == socket.AF_INET6
                    # ...FreeBSD...
                    and _osname == 'freebsd'
                    # ...if the address is link-local (fe80::smth)...
                    and ifAddrsC.contents.addr.contents.data[6] == 0xfe
                    and ifAddrsC.contents.addr.contents.data[7] == 0x80
                    # ...stores the scopeid in the 3-4 bytes
                    and ifAddrsC.contents.addr.contents.data[8:10] != [0, 0]
                ):
                    # YOU DON'T WANT TO KNOW HOW MANY KITTENS THAT BASTARDS KILLED
                    scopeid = (ifAddrsC.contents.addr.contents.data[8] << 8) | ifAddrsC.contents.addr.contents.data[9]
                    ifAddrsC.contents.addr.contents.data[8] = 0
                    ifAddrsC.contents.addr.contents.data[9] = 0
                addresses.append(
                    IfAddr(
                        ifAddrsC.contents.name,
                        family,
                        getNameInfo(ifAddrsC.contents.addr),
                        getNameInfo(ifAddrsC.contents.netmask),
                        getNameInfo(ifAddrsC.contents.broadaddr),
                        ifAddrsC.contents.flags,
                        scopeid=scopeid
                    )
                )
            ifAddrsC = ifAddrsC.contents.next
    finally:
        if ifAddrs:
            libc().freeifaddrs(ifAddrs)

    return result


class __SockAddr(ctypes.Structure):
    _fields_ = [
        ("lenFamily1", ctypes.c_ubyte),
        ("lenFamily2", ctypes.c_ubyte),
        ("data", ctypes.c_ubyte * 14),
    ]


class _InAddr(ctypes.Structure):
    _fields_ = []


class __SockAddrIn(ctypes.Structure):
    _fields_ = [
        ("lenFamily1", ctypes.c_ubyte),
        ("lenFamily2", ctypes.c_ubyte),
        ("port", ctypes.c_ushort),
        ("addr", _InAddr),
        # something else follows
    ]


class __SockAddrIn6(ctypes.Structure):
    _fields_ = [
        ("lenFamily1", ctypes.c_ubyte),
        ("lenFamily2", ctypes.c_ubyte),
        ("port", ctypes.c_ushort),
        ("flowinfo", ctypes.c_uint32),
        ("addr", _InAddr),
        # something else follows
    ]


class __IfAddrs(ctypes.Structure):
    pass


__IfAddrs._fields_ = [
    ("next", ctypes.POINTER(__IfAddrs)),
    ("name", ctypes.c_char_p),
    ("flags", ctypes.c_uint),
    ("addr", ctypes.POINTER(__SockAddr)),
    ("netmask", ctypes.POINTER(__SockAddr)),
    ("broadaddr", ctypes.POINTER(__SockAddr)),
]
