import os
import socket
import ctypes
import errno
import struct

import six

__all__ = [
    "getAddrRepr",
    "packIP",
    "unpackIP",
    "getFreePort",
]

if six.PY3:
    long = int

if os.name == "posix":
    inet_pton = socket.inet_pton
    inet_ntop = socket.inet_ntop
else:
    ws2 = ctypes.windll.ws2_32  # getLib(['ws2_32.dll'], True)
    inet_pton_ = ws2.inet_pton
    # inet_pton_.argtypes = [ctypes.c_int, ctypes.c_char_p, ctypes.c_char_p]
    inet_ntop_ = ws2.inet_ntop
    # inet_ntop_.argtypes = [ctypes.c_int, ctypes.c_int, ctypes.c_int, ctypes.c_size_t]

    def inet_pton(family, ip):
        sz = 4 if family == socket.AF_INET else 16
        res = ctypes.create_string_buffer('', sz)
        ret = inet_pton_(family, ip, res)
        if not ret:
            raise EnvironmentError(errno.EINVAL, 'Illegal IP address string passed to inet_pton')
        elif ret != 1:
            errcode = ws2.WSAGetLastError()
            raise OSError(errcode, errno.errorcode.get(errcode, 'Unknown error'))
        return res[:]  # .strip('\x00')

    def inet_ntop(family, packedIp):
        sz = 16 if family == socket.AF_INET else 46
        res = ctypes.create_string_buffer('', sz)
        # print ctypes.c_char_p(packedIp)
        ret = inet_ntop_(family, packedIp, res, sz)
        if not ret:
            errcode = ws2.WSAGetLastError()
            raise OSError(errcode, errno.errorcode.get(errcode, 'Unknown error'))
        return res[:]  # .strip('\x00')


def getAddrRepr(addr, family=socket.AF_INET):
    if family in (socket.AF_INET, socket.AF_INET6):
        try:
            host = socket.getfqdn(addr[0])
        except socket.error:
            host = addr[0]
        return "{0}:{1}".format(host, addr[1])
    else:
        return str(addr)


__ipv6Struct = struct.Struct('>QQ')
__ipv4Struct = struct.Struct('>I')


def packIP(s, family=socket.AF_INET):
    """
    Pack ip in integer
    :param str s: ip address ('127.0.0.1' / '::1' etc.)
    :return: ip address as integer
    :rtype: long
    """
    if family == socket.AF_INET6:
        pos = s.find("%")
        if pos >= 0:
            s = s[:pos]
    s = inet_pton(family, s)
    if family == socket.AF_INET:
        return long(__ipv4Struct.unpack(s)[0])
    elif family == socket.AF_INET6:
        result = __ipv6Struct.unpack(s)
        return long((result[0] << 64) + result[1])

__ipv6LowMask = (1 << 64) - 1


def unpackIP(i, family=socket.AF_INET):
    """
    Unpack ip from integer
    :param i: packed ip
    :type i: int or long
    :return: ip address ('127.0.0.1' / '::1' etc.)
    """
    if family == socket.AF_INET:
        s = __ipv4Struct.pack(i)
    else:
        s = __ipv6Struct.pack(i >> 64, i & __ipv6LowMask)
    return inet_ntop(family, s).strip('\x00')


def getFreePort(families=None):
    """
    Returns some available port
    :param families: address family or collection of afs
    """
    if families is None:
        families = [socket.AF_INET]
    if isinstance(families, int):
        families = [families]
    families = set(families)
    startAf = families.pop()
    result = None
    while result is None:
        sock = None

        try:
            sock = socket.socket(startAf)
            sock.bind(('', 0))
            result = sock.getsockname()[1]
        finally:
            if sock:
                sock.close()

        for af in families:
            if result is None:
                continue
            try:
                sock = socket.socket(af)
                sock.bind(('', result))
            except EnvironmentError:
                result = None
            finally:
                sock.close()

    return result
