import fcntl
import os
import errno
import sys
import six
import gevent
import gevent.socket


class Pipe(object):
    def __init__(self, r=None, w=None):
        self.r = r
        self.w = w
        self.setBlocking()

    def setBlocking(self, blocking=True):
        self.blocking = blocking
        if self.r:
            curFlags = fcntl.fcntl(self.r, fcntl.F_GETFL)
            fcntl.fcntl(self.r, fcntl.F_SETFL, curFlags & ~os.O_NONBLOCK if blocking else curFlags | os.O_NONBLOCK)
        if self.w:
            curFlags = fcntl.fcntl(self.w, fcntl.F_GETFL)
            fcntl.fcntl(self.w, fcntl.F_SETFL, curFlags & ~os.O_NONBLOCK if blocking else curFlags | os.O_NONBLOCK)

    def open(self):
        if self.r or self.w:
            return
        self.r, self.w = os.pipe()
        self.setBlocking(self.blocking)

    def _read(self, l):
        while True:
            try:
                return os.read(self.r, l)
            except EnvironmentError as err:
                if err[0] == errno.EBADF:
                    return ''
                if err[0] != errno.EWOULDBLOCK:
                    raise
                sys.exc_clear()
            try:
                gevent.socket.wait_read(self.r, None)
            except EnvironmentError as err:
                if err[0] == errno.EBADF:
                    return ''
                raise

    read = _read

    def _write(self, d):
        try:
            return os.write(self.w, d)
        except EnvironmentError as err:
            if err[0] != errno.EWOULDBLOCK:
                raise
            sys.exc_clear()
            try:
                gevent.socket.wait_write(self.w)
            except EnvironmentError as err:
                if err[0] == errno.EBADF:
                    return 0
                raise
            try:
                return os.write(self.w, d)
            except EnvironmentError as err:
                if err[0] == errno.EWOULDBLOCK:
                    return 0
                raise

    write = _write

    def readExact(self, l):
        read = []
        readLen = 0
        while readLen < l:
            read.append(self._read(l - readLen))
            if not read[-1]:
                raise IOError(errno.ECONNRESET, "Broken pipe")
            readLen += len(read[-1])
        return "".join(read)

    def writeAll(self, d):
        dataSent = 0
        if not isinstance(d, six.binary_type):
            d = six.b(d)
        view = memoryview(d)
        while dataSent < len(d):
            dataSent += self._write(view[dataSent:])

    def close(self):
        self.closeR()
        self.closeW()

    def closeR(self):
        if self.r:
            os.close(self.r)
            self.r = None

    def closeW(self):
        if self.w:
            os.close(self.w)
            self.w = None

    def __enter__(self):
        self.open()
        return self

    def __exit__(self, _excType, _excValue, _traceback):
        self.close()
