"""
Tests for socket server
"""

import socket
import gevent.socket
import gevent.event

from kernel.util.unittest import TestCase, main
from kernel.util.net.socketserver import getValidAddrs, getInvalidAddrs, SocketServer
from kernel.util.net.getifaddrs import IfAddr


class TestSocketServer(TestCase):
    """Tests for socket server"""
    def test_filter_ipv6_addrs(self):
        families = [gevent.socket.AF_INET, gevent.socket.AF_INET6]
        ifAddrs = {
            'lo': [
                IfAddr('lo', gevent.socket.AF_INET, '127.0.0.1', '255.0.0.0', '127.0.0.1', 65609, 0),
                IfAddr('lo', gevent.socket.AF_INET6, '::1', 'ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff', None, 65609, 0),
            ],
            'vlan763': [
                IfAddr('vlan763', gevent.socket.AF_INET6, '2a02:6b8:0:a1f::5d9e:a8cb', 'ffff:ffff:ffff:ffff::', None, 69699, 0),
                IfAddr('vlan763', gevent.socket.AF_INET6, 'fe80::96de:80ff:fe81:8ea%vlan763', 'ffff:ffff:ffff:ffff::', None, 69699, 0),
            ],
            'eth0': [
                IfAddr('eth0', gevent.socket.AF_INET, '93.158.168.203', '255.255.252.0', '93.158.171.255', 69699, 0),
                IfAddr('eth0', gevent.socket.AF_INET6, '2a02:6b8:0:1a02::5d9e:a8cb', 'ffff:ffff:ffff:ffff::', None, 69699, 0),
                IfAddr('eth0', gevent.socket.AF_INET6, 'fe80::96de:80ff:fe81:8ea%eth0', 'ffff:ffff:ffff:ffff::', None, 69699, 0),
            ],
        }
        interfaces = getValidAddrs(families, ifAddrs)
        for addrs in ifAddrs.values():
            for addr in addrs:
                self.assertTrue(addr in interfaces, '{} is absent'.format(addr))

    def test_invalid_addrs(self):
        families = [gevent.socket.AF_INET, gevent.socket.AF_INET6]
        ifAddrs = {
            'igb0': [
                IfAddr('igb0', gevent.socket.AF_INET, '84.201.134.2', '255.255.254.0', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET6, 'fe80::225:90ff:fe0d:ed2', 'ffff:ffff:ffff:ffff::', None, 65609, 1),
                IfAddr('igb0', gevent.socket.AF_INET6, '2a02:6b8:0:149f::54c9:8602', '', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET, '84.201.134.37', '255.255.255.255', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET6, '2a02:6b8:0:149f::54c9:8625', '', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET, '84.201.134.72', '255.255.255.255', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET6, '2a02:6b8:0:149f::54c9:8648', '', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET, '84.201.134.107', '255.255.255.255', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET6, '2a02:6b8:0:149f::54c9:866b', '', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET, '84.201.134.142', '255.255.255.255', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET6, '2a02:6b8:0:149f::54c9:868e', '', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET, '84.201.134.177', '255.255.255.255', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET6, '2a02:6b8:0:149f::54c9:86b1', '', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET, '84.201.134.212', '255.255.255.255', None, 65609, 0),
                IfAddr('igb0', gevent.socket.AF_INET6, '2a02:6b8:0:149f::54c9:86d4', '', None, 65609, 0),
            ]
        }
        validAddrs = {
            IfAddr('igb0', gevent.socket.AF_INET, '84.201.134.2', '255.255.254.0', None, 65609, 0),
            IfAddr('igb0', gevent.socket.AF_INET6, 'fe80::225:90ff:fe0d:ed2', 'ffff:ffff:ffff:ffff::', None, 65609, 1),
        }
        interfaces = getInvalidAddrs(families, ifAddrs)
        for interface in interfaces:
            self.assertFalse(interface in validAddrs)

    def test_simple_listen(self):
        """Simple listen test"""
        connectedEvent = gevent.event.Event()
        receivedEvent = gevent.event.Event()
        data = 'some random data 3102-3913-19301' * 1024
        receivedData = []

        def _newConnection(socket, _):
            connectedEvent.set()
            received = 0
            while received < len(data):
                receivedData.append(socket.recv(1024))
                received += len(receivedData[-1])
            receivedEvent.set()

        with SocketServer(0, _newConnection) as server:
            family = socket.AF_INET6 if socket.has_ipv6 else socket.AF_INET
            sock = gevent.socket.socket(family)
            sock.connect(gevent.socket.getaddrinfo("localhost", server.addressSpec, family, socket.SOCK_STREAM)[0][4])
            connectedEvent.wait()
            sock.sendall(data)
            receivedEvent.wait()
            self.assertEqual(data, "".join(receivedData))

if __name__ == '__main__':
    main()
