# -*- coding: utf-8 -*-
"""
High-level wrapper over urllib2 and pkg_resources, that can fetch data from the following URI types:
    - pkg_resources:package@path
    - pkg_resources:path
    - file:path
    - http://domain/path
    - https://domain/path
    - about:blank
"""

__author__ = 'torkve'

try:
    from urllib2 import splittype, splituser
except ImportError:
    try:
        from urllib import splittype, splituser
    except ImportError:
        from urllib.parse import splittype, splituser


def load_pkg_resources(name, default_package=None, as_stream=False, **kwargs):
    from pkg_resources import resource_string, resource_stream
    name = splittype(name)[1]
    _package, name = splituser(name)
    _package = _package or default_package

    if _package is None:
        raise RuntimeError("No package name in URI and no package specified")

    if as_stream:
        return resource_stream(_package, name)
    else:
        return resource_string(_package, name)


def load_file(path, as_stream=False, **kwargs):
    path = splittype(path)[1]
    f = open(path, 'rb')
    if as_stream:
        return f
    else:
        return f.read()


def load_http(url, as_stream=False, **kwargs):
    from pkg_resources import require
    from cStringIO import StringIO
    require('requests')
    import requests

    req = requests.get(url)
    if req.status_code != requests.codes.ok:
        raise RuntimeError("Cannot fetch URI: HTTP {}".format(req.status_code))

    if as_stream:
        return StringIO(req.text())  # Current skynet 'requests' package doesn't support streams
    else:
        return req.text()


def load_about(url, **kwargs):
    path = splittype(url)[1]
    if path != 'blank':
        raise RuntimeError("Unknown URI in about: {}".format(url))
    return None


_schemes = {
    'pkg_resources': load_pkg_resources,
    'file': load_file,
    'http': load_http,
    'https': load_http,
    'about': load_about,
}


def load(url, default_package=None, as_stream=False):
    """
    Load target @c url
    :param string default_package: package to look in if applicable.
                                   Used by @c pkg_resources scheme to read urls without package_name
    :param bool as_stream:         if set, result will be returned as stream rather than string
    :return:                       None or string or stream
    :raise RuntimeError:           if url cannot be loaded
    :raise ImportError:            if pkg_resources url requires unknown package
    """
    scheme = splittype(url)[0]
    if scheme is None or scheme not in _schemes:
        raise RuntimeError("Unknown URI scheme: {}".format(scheme))
    return _schemes[scheme](url, default_package=default_package, as_stream=as_stream)
