# Import everything here, because we want ability to make this:
# >>> from kernel.util import pickle
# >>> try:
# ...     s = pickle.dumps('some data')
# ... except pickle.PicklingError:
# ...     pass  # or do something useful

from __future__ import absolute_import
import marshal
import sys
import six
try:
    import cPickle
    from cPickle import *  # noqa
except ImportError:
    import pickle as cPickle
    from pickle import *  # noqa
import importlib
from functools import partial
copy_reg = six.moves.copyreg
import types
import errno

__sys_modules = sys.modules
__unpickle_overrides = {}


def string_of_errno(errnum):
    return errno.errorcode[errnum]


def errno_of_string(errstr):
    try:
        return getattr(errno, errstr)
    except AttributeError:
        return None


def register_unpickle_override(module, name, callback):
    """
    Register some callable (function, or class with __call__() property)
    to be called for unpickling the type ::name from from ::module

    :param module: name of the module, containing the type
    :param name: name of the type
    :param callback: some callable to execute for the type
    """
    __unpickle_overrides[(module, name)] = callback


def __unpickle_global(module, name):
    """
    Support function for Unpickler be able to detect the
    unpickle method for the ::name from ::module
    First it checks installed overrides, than looks for
    the ::name in standard modules.
    :param module: name of the module, containing the type
    :param name: name of the type
    :return: some callable
    """
    unpickler = __unpickle_overrides.get((module, name))
    if unpickler is not None:
        return unpickler
    mod = __sys_modules.get(module)
    if mod is not None:
        return getattr(mod, name)
    __import__(module)
    mod = __sys_modules[module]
    return getattr(mod, name)


def __unpickler(*kargs, **kwargs):
    """
    Method returning Unpickler object with the
    find_global property properly set
    """
    unpickler = cPickle.Unpickler(*kargs, **kwargs)
    unpickler.find_global = __unpickle_global
    return unpickler


def _unpickle_environmenterror(exception_type, *kargs, **kwargs):
    result = exception_type(*kargs, **kwargs)
    if result.errno is not None and isinstance(result.errno, six.string_types):
        errno = errno_of_string(result.errno)
        result.errno = errno
        result.args = (errno, result.strerror)
    return result


def _unpickle_lambda(name, argdefs, imports, codumped):
    globs = {
        modname: importlib.import_module(mod)
        for modname, mod in six.iteritems(imports)
    }
    code = marshal.loads(codumped)
    return types.FunctionType(code, globs, name, argdefs)


def _pickle_method(method):
    # TODO: python>=3, <3.3 support
    # TODO: python>=3.3 support with __qualname__

    methodName = method.__func__.__name__

    if methodName.startswith('__') and not methodName.endswith('__'):
        # No im_class in python 3 (method.__self__.__class__)
        for cls in method.im_class.mro():
            try:
                testName = '_{0}{1}'.format(cls.__name__, methodName)
                if getattr(cls, testName).__func__ == method.__func__:
                    methodName = testName
            except AttributeError:
                pass
    self = method.__self__

    # Unbound method (python 2)
    if self is None:
        self = method.im_class

    return getattr, (self, methodName)


def _pickle_lambda(fun):
    name = fun.__name__
    if name != '<lambda>':
        raise cPickle.PicklingError("Can't pickle function object: {}".format(name))

    if fun.__closure__ is not None:
        raise cPickle.PicklingError("Can't pickle closures, only clean lambda allowed: {}".format(name))

    co = fun.__code__
    imports = {
        mn: mod.__name__
        for mn, mod in list(fun.__globals__.items())

        if mn not in co.co_freevars
        and mn not in co.co_varnames
        and mn in co.co_names
        and isinstance(mod, types.ModuleType)
    }
    args = (
        name,
        fun.__defaults__,
        imports,
        marshal.dumps(co),
    )
    return _unpickle_lambda, args


def _pickle_environmenterror(error):
    if error.errno is not None and isinstance(error.errno, int):
        errstr = string_of_errno(error.errno)
        error.args = (errstr, error.strerror)
        error.errno = errstr
    return error.__reduce__()

copy_reg.pickle(types.MethodType, _pickle_method)
# cPickle always tries simple function pickling first, and only then falls back to __reduce__ with copy_reg,
# so we can safely register override for function type
copy_reg.pickle(types.LambdaType, _pickle_lambda)
copy_reg.pickle(EnvironmentError, _pickle_environmenterror)
copy_reg.pickle(OSError, _pickle_environmenterror)
copy_reg.pickle(IOError, _pickle_environmenterror)

register_unpickle_override('exceptions', 'EnvironmentError', partial(_unpickle_environmenterror, EnvironmentError))
register_unpickle_override('exceptions', 'OSError', partial(_unpickle_environmenterror, OSError))
register_unpickle_override('exceptions', 'IOError', partial(_unpickle_environmenterror, IOError))
Unpickler = __unpickler

dumps = partial(cPickle.dumps, protocol=cPickle.HIGHEST_PROTOCOL)
dump = partial(cPickle.dump, protocol=cPickle.HIGHEST_PROTOCOL)
Pickler = partial(cPickle.Pickler, protocol=cPickle.HIGHEST_PROTOCOL)
